//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.usability2.rules.patterns;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.keyboardmaps.VirtualKey;
import de.ugoe.cs.autoquest.usability.EvaluationMethodCaller;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescriptionResolver;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRule;
import de.ugoe.cs.autoquest.usability.rules.UsabilityUsageProblem;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.Conjunction;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.Follows;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.IFilter;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.Label;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.filter.AnyFilter;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.filter.EventTypeFilter;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.filter.KeyPressFilter;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.filter.TaskTypeFilter;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.visitors.FindAtStart;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.visitors.FindContained;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.wrapper.ITaskEntry;
import de.ugoe.cs.autoquest.plugin.usability2.rules.results.IMatch;
import de.ugoe.cs.autoquest.plugin.usability2.rules.results.IResult;
import de.ugoe.cs.autoquest.plugin.usability2.tools.TaskUtilities;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * <p>
 * Test Rule to see if new pattern method may be used for problem checking
 * </p>
 * 
 * @author Konni Hartmann
 */
public class PatternProblem extends UsabilityRule implements UsabilityUsageProblem {

    private IFilter tab_pattern;
    private IFilter scroll_click_pattern;

    /**
     * <p>
     * TODO: comment
     * </p>
     * 
     * @param taskTree
     */
    public PatternProblem(ITaskModel taskModel) {
        super(taskModel);
        this.name = "PatternProblem";
        this.defect =
            new UsabilityProblemDescriptionResolver().descriptionFor(this.getClass()
                .getSimpleName());
        initUsagePattern();
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     * 
     */
    private void initUsagePattern() {
        this.tab_pattern = 
            new Follows(new Label("target", new FindContained(new KeyPressFilter(VirtualKey.TAB))),
                        new Label("follower", new FindAtStart(EventTypeFilter.TEXT_INPUT)));
//                        new Label("follower", new FindAtStart(EventTypeFilter.MOUSE_CLICK)));

//        this.pattern2 =
//            new FindContained(new Conjunction(TaskTypeFilter.ITERATION,
//                                              new FindContained(EventTypeFilter.TEXT_INPUT),
//                                              new FindContained(EventTypeFilter.MOUSE_CLICK)));

        this.scroll_click_pattern =
            new Follows(new Label("source", new FindContained(EventTypeFilter.SCROLL)),
                        new Label("target", new FindAtStart(EventTypeFilter.MOUSE_CLICK)));

    }

    private class Statistics implements Comparable<Statistics> {
        Set<ITask> all = new HashSet<ITask>();
        HashMap<ITask, ITask> match = new HashMap<ITask, ITask>();

        private int getCount(ITask task) {
            if (task instanceof ITaskEntry)
                task = ((ITaskEntry) task).getReference();
            return taskModel.getTaskInfo(task).getCount();
        }
        
        void addEqual(ITask task) {
            all.add(task);
        }

        int matchCount() {
            int match = 0;
            for (ITask task : this.match.values()) {
                match += getCount(task);
            }
            return match;
        }

        int allCount() {
            int all = 0;
            for (ITask task : this.all) {
                all += getCount(task);
            }
            return all;
        }

        float percent() {
            return ((float) matchCount()) / allCount();
        }

        @Override
        public int compareTo(Statistics o) {
            float thisPcnt = percent();
            float otherPcnt = o.percent();

            if (thisPcnt < otherPcnt)
                return -1;
            if (thisPcnt > otherPcnt)
                return 1;
            return 0;
        }

        @Override
        public String toString() {
            StringBuilder str = new StringBuilder();
            
            for (ITask task : all) {
                if(task instanceof ITaskEntry)
                    task = ((ITaskEntry) task).getReference();
                if(task instanceof IEventTask) {
                    IEventTaskInstance next = (IEventTaskInstance) task.getInstances().iterator().next();
                    str.append(next.getEvent().getTarget().getStringIdentifier()).append(" [");
                    str.append(task.getId()).append(']');
                }
                else
                    str.append(task.getDescription());
                str.append(", ");
            }

            str.append(" :\n  ");
            
            for (ITask task : match.values()) {
                str.append(task).append(", ");
            }

            str.append('(').append(percent()).append(')');

            return str.toString();
        }

        public void addMatch(ITask source, ITask task) {
            ITask maxTask = match.get(source);
            if (maxTask == null || getCount(maxTask) < getCount(task)) {
                maxTask = task;
                match.put(source, maxTask);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usability.rules.UsabilityRule#check()
     */
    @Override
    public Optional<UsabilityProblemDescription> check() {
        Optional<UsabilityProblemDescription> present = Optional.absent();

        System.out.println();
        System.out.printf("(PP) Starting analysis on %s\n", this.tab_pattern);

        Collection<ITask> tasks = TaskUtilities.findRootTasks(taskModel.getTasks());

        System.out.println("--");
        System.out.println("PATTERN3:");

        Map<ITask, Statistics> results = new HashMap<ITask, Statistics>();

        for (ITask task : taskModel.getTasks()) {
            IResult result = this.scroll_click_pattern.match(task);

            if (result.isPresent()) {
                present = Optional.of(this.defect);

                for (IMatch match : result) {
                    ITask target = match.getLabeledResults().get("target");
                    ITask source = match.getLabeledResults().get("source");

                    Statistics stats = results.get(target);
                    if (stats == null) {
                        stats = new Statistics();
                        stats.addEqual(target);
                        results.put(target, stats);
                    }
                    stats.addMatch(source, task);
                }
                

                /*
                 * int cnt = task.getInstances().size(); for (ITask oTask : taskModel.getTasks()) {
                 * if (!oTask.equals(task)) { TaskEquality equality =
                 * TaskEqualityRuleManager.getInstance().compare(task, oTask); if
                 * (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL)) { cnt +=
                 * oTask.getInstances().size();
                 * 
                 * stats.addEqual(task); results.put(oTask, stats); } } }
                 */
                /*
                 * System.out.printf("3: %s, %s, %d/%d\n", task.getId(), task.getDescription(), task
                 * .getInstances().size(), target.getInstances().size());
                 */
            }

        }
        analyzeResults(results);

        /*
         * System.out.println("PATTERN2:"); for (ITask task : tasks) { IResult result =
         * this.pattern2.match(task);
         * 
         * result = this.pattern2.match(task);
         * 
         * if (result.isPresent()) { present = Optional.of(this.defect);
         * System.out.printf("2: %s, %s, %d\n", task.getId(), task.getDescription(), task
         * .getInstances().size()); } }
         */
        System.out.println("--");
        System.out.println("PATTERN1:");

        for (ITask task : tasks) {
            IResult result = this.tab_pattern.match(task);

            if (result.isPresent()) {
                present = Optional.of(this.defect);
                System.out.printf("1: %s, %s, %d\n", task.getId(), task.getDescription(), task
                    .getInstances().size());
                for (IMatch match : result) {
                    ITask target = match.getLabeledResults().get("target");
                    ITask follower = match.getLabeledResults().get("follower");
                    System.out.printf(" - %s -> %s\n", target.getDescription(), follower.getDescription()); 
                }
            }

        }

        System.out.println("Finished Pattern analysis");
        return present;
    }

    private void analyzeResults(Map<ITask, Statistics> results) {

        Set<Statistics> stats = new HashSet<Statistics>(results.values());

        List<Statistics> pcnt95 = new LinkedList<Statistics>();
        List<Statistics> pcnt75 = new LinkedList<Statistics>();
        List<Statistics> pcnt50 = new LinkedList<Statistics>();
        List<Statistics> pcnt25 = new LinkedList<Statistics>();
        List<Statistics> pcnt05 = new LinkedList<Statistics>();

        for (Statistics s : stats) {
            float percent = s.percent();

            if (percent >= 0.95)
                pcnt95.add(s);
            else if (percent >= 0.75)
                pcnt75.add(s);
            else if (percent >= 0.50)
                pcnt50.add(s);
            else if (percent >= 0.25)
                pcnt25.add(s);
            else if (percent >= 0.05)
                pcnt05.add(s);
        }

        Comparator<Statistics> reverse = new Comparator<Statistics>() {
            @Override
            public int compare(Statistics o1, Statistics o2) {
                return o2.matchCount() - o1.matchCount();
            }
        };

        Collections.sort(pcnt95, reverse);
        Collections.sort(pcnt75, reverse);
        Collections.sort(pcnt50, reverse);
        Collections.sort(pcnt25, reverse);
        Collections.sort(pcnt05, reverse);

        printFirstN(pcnt95, 3, 0.95);
        printFirstN(pcnt75, 3, 0.75);
        printFirstN(pcnt50, 3, 0.50);
        printFirstN(pcnt25, 3, 0.25);
        printFirstN(pcnt05, 3, 0.05);
    }

    private void printFirstN(List<Statistics> list, int i, double margin) {
        if (!list.isEmpty()) {
            System.out.printf(">= %f :\n", margin);
            for (int j = 0; j < list.size() && j < i; j++) {
                Statistics s = list.get(j);
                System.out.printf("%d [%d]: %s\n", j + 1, s.matchCount(), s);
            }
            if (list.size() >= i) {
                Statistics s = list.get(list.size() - 1);
                System.out.printf("%d [%d]: %s\n", list.size() - 1, s.matchCount(), s);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.autoquest.usability.rules.UsabilityRule#callEvaluationMetho(de.ugoe.cs.autoquest
     * .usability.EvaluationMethodCaller)
     */
    @Override
    public Optional<UsabilityProblemDescription> callEvaluationMethod(EvaluationMethodCaller evaluationMethodCaller)
    {
        return evaluationMethodCaller.check(this);
    }
}
