//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.htmlmonitor;

/**
 * <p>
 * represents a document on a web server including its path and query in the URL.
 * </p>
 * 
 * @author Patrick Harms
 */
class HtmlDocument extends HtmlGUIElement {

    /**
     * the server on which the document resists
     */
    private HtmlServer server;
    
    /**
     * the path to the document on the server
     */
    private String path;
    
    /**
     * the query on the path that created the document (including the leading questionmark)
     */
    private String query;
    
    /**
     * the title of the document, null if none is present
     */
    private String title;
    
    /**
     * the hash code of this document
     */
    private int hashCode;
    
    /**
     * <p>
     * instantiates a new document element
     * </p>
     *
     * @param id     the id of the document
     * @param server the server on which the document resists
     * @param path   the path to the document on the server
     * @param query  the query on the path that created the document (may include the questionmark)
     * @param title  the title of the document, null if none is present
     */
    HtmlDocument(String id, HtmlServer server, String path, String query, String title) {
        super(id, server);

        if (server == null) {
            throw new IllegalArgumentException("server must not be null");
        }

        if (path == null) {
            throw new IllegalArgumentException("path must not be null");
        }

        this.server = server;
        this.path = path;
        this.query = query;
        this.title = title;
        
        if ((this.query != null) && (!this.query.startsWith("?"))) {
            this.query = "?" + this.query;
        }
        
        this.hashCode = this.server.hashCode() + this.path.hashCode() +
            (this.query != null ? this.query.hashCode() : 0) +
            (this.title != null ? this.title.hashCode() : 0);
    }

    /**
     * @return the server on which the document resists
     */
    HtmlServer getServer() {
        return server;
    }

    /**
     * @return the path to the document on the server
     */
    String getPath() {
        return path;
    }

    /**
     * @return the query on the path that created the document (including the leading questionmark)
     */
    String getQuery() {
        return query;
    }

    /**
     * @return the title of the document, null if none is present
     */
    String getTitle() {
        return title;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        else if (obj instanceof HtmlDocument) {
            return equals((HtmlDocument) obj);
        }
        else {
            return false;
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.htmlmonitor.HtmlGUIElement#equals(de.ugoe.cs.autoquest.htmlmonitor.HtmlGUIElement)
     */
    public boolean equals(HtmlDocument other) {
        if (this == other) {
            return true;
        }

        return (server.equals(other.server) && path.equals(other.path) &&
                (query != null ? query.equals(other.query) : other.query == null) &&
                (title != null ? title.equals(other.title) : other.title == null));
    }

    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return hashCode;
    }

}