//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.guimodel;

import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec;

/**
 * <p>
 * a GUI element specification of an HTML document, i.e., a page on an HTML server. This is the
 * element for a GUI model of an HTML web site being always and only the children of servers. It
 * is identified through the server on which it resists, its path, a potential query, and a title.
 * Its children are HTML page elements
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTMLDocumentSpec extends HTMLGUIElementSpec implements IGUIElementSpec {

    /**
     * <p>
     * default serial version UID
     * </p> 
     */
    private static final long serialVersionUID = 1L;
    
    /**
     * <p>
     * the server on which the document resists
     * </p>
     */
    private HTMLServerSpec server;
    
    /**
     * <p>
     * the path in the URL of the document
     * </p>
     */
    private String path;
    
    /**
     * <p>
     * the query in the URL of the document
     * </p>
     */
    private String query;
    
    /**
     * <p>
     * the title of the document
     * </p>
     */
    private String title;
    
    /**
     * <p>
     * initializes the document with its server, path, query, and title
     * </p>
     *
     * @param server the server on which the document resists
     * @param path   the path in the URL of the document
     * @param query  the query in the URL of the document
     * @param title  the title of the document
     * 
     * @throws IllegalArgumentException if the server or path are invalid, i.e. null
     */
    public HTMLDocumentSpec(HTMLServerSpec server, String path, String query, String title) {
        super("document");
        
        if (server == null) {
            throw new IllegalArgumentException("server must not be null");
        }
        else if (path == null) {
            throw new IllegalArgumentException("pagePath must not be null");
        }
        
        this.server = server;
        this.path = path;
        this.query = query;
        this.title = title;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec#getSimilarity(IGUIElementSpec)
     */
    @Override
    public boolean getSimilarity(IGUIElementSpec other) {
        if (other instanceof HTMLDocumentSpec) {
            HTMLDocumentSpec otherSpec = (HTMLDocumentSpec) other;
            
            if (!super.getSimilarity(otherSpec)) {
                return false;
            }
            else if (!server.getSimilarity(otherSpec.server)) {
                return false;
            }
            else if (!path.equals(otherSpec.path)) {
                return false;
            }
            else if (query != null ? !query.equals(otherSpec.query) : otherSpec.query != null) {
                return false;
            }
            else {
                return (title != null ? title.equals(otherSpec.title) : otherSpec.title == null);
            }
        }
        
        return false;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "Document(" + getPath() + ", \"" + getTitle() + "\")";
    }

    /**
     * <p>
     * returns the server on which the document resists
     * </p>
     *
     * @return the server on which the document resists
     */
    public HTMLServerSpec getServer() {
        return server;
    }

    /**
     * <p>
     * returns the path in the URL of the document
     * </p>
     *
     * @return the path in the URL of the document
     */
    String getPath() {
        return path;
    }

    /**
     * <p>
     * returns the query in the URL of the document
     * </p>
     *
     * @return the query in the URL of the document
     */
    String getQuery() {
        return query;
    }

    /**
     * <p>
     * returns the title of the document
     * </p>
     *
     * @return the title of the document
     */
    String getTitle() {
        return title;
    }

}
