//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.httpmonitor.proxy;

import javax.servlet.Servlet;

import de.ugoe.cs.autoquest.httpmonitor.HttpMonitorComponent;
import de.ugoe.cs.autoquest.httpmonitor.HttpMonitorException;
import de.ugoe.cs.autoquest.httpmonitor.HttpMonitorExchangeHandler;
import de.ugoe.cs.autoquest.httpmonitor.HttpMonitorLogManager;
import de.ugoe.cs.autoquest.httpmonitor.HttpMonitorServer;
import de.ugoe.cs.autoquest.httpmonitor.Runner;
import de.ugoe.cs.autoquest.httpmonitor.ShutdownHook;
import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * TODO comment
 * The HTML monitor starts a web server ({@link HttpMonitorServer}) that receives log messages
 * of the HTML monitor java script. These messages are logged using the
 * {@link ExchangeListenerManager}. The class assures that on shutdown e.g. caused by CTRL-C the
 * server and the log manager are stopped correctly.
 * </p>
 * 
 * @author Patrick Harms
 */
public class HttpMonitoringProxy implements HttpMonitorComponent {

    /**
     * the port on which the webserver shall listen.
     */
    private int port;
    
    /**
     * the web server receiving the log messages
     */
    private HttpMonitorServer server;
    
    /**
     * the web server receiving the log messages
     */
    private ExchangeListenerManager exchangeListenerManager;
    
    /**
     * 
     */
    private HttpMonitorExchangeHandler exchangeHandler;
    
    /**
     * the directory into which the log files shall be written
     */
    private String logFileBaseDir;

    /**
     * the thread needed to handle CTRL-C events
     */
    private Thread shutdownHook;

    /** */
    private String proxiedServer;
    
    /** */
    private int proxiedPort;
    
    /** */
    private String httpMonitorServer;
    
    /** */
    private int httpMonitorPort;
    
    /**
     * <p>
     * initializes the monitor with the command line arguments. Those may be the log directory
     * as first argument and the port to listen on as second
     * </p>
     *
     * @param commandLineArguments the command line arguments when starting the monitor using
     *                             the {@link Runner}
     */
    public HttpMonitoringProxy(String[] commandLineArguments) {
        if (commandLineArguments.length != 4) {
            Console.printerrln("invalid number of command line parameters. Three are required:");
            Console.printerrln("  1. the port on which to listen");
            Console.printerrln("  2. the server and port to be proxied (format \"host:port\")");
            Console.printerrln
                ("  3. variant a: the value \"local\" for logging recorded exchanges locally");
            Console.printerrln
                ("     variant b: the server and port of the HTTP monitor to send the recorded\n" +
                 "                exchanges to (format \"host:port\")");
            throw new IllegalArgumentException();
        }
        
        this.logFileBaseDir = commandLineArguments[0];

        try {
            this.port = Integer.parseInt(commandLineArguments[1]);
        }
        catch (NumberFormatException e) {
            Console.printerrln("invalid port specification " + commandLineArguments[1]);
        }
        Console.println("listening on port " + this.port);

        proxiedServer = commandLineArguments[2];
        
        int index = proxiedServer.indexOf(':');
        if (index > -1) {
            try {
                proxiedPort = Integer.parseInt(proxiedServer.substring(index + 1));
            }
            catch (NumberFormatException e) {
                Console.printerrln
                    ("invalid port specification " + proxiedServer.substring(index + 1));
            }
            
            proxiedServer = proxiedServer.substring(0, index);
        }
        else {
            proxiedPort = 80;
        }
        
        Console.println("proxing " + proxiedServer + ":" + proxiedPort);
        
        httpMonitorServer = commandLineArguments[3];
        
        index = httpMonitorServer.indexOf(':');
        if (index > -1) {
            try {
                httpMonitorPort = Integer.parseInt(httpMonitorServer.substring(index + 1));
            }
            catch (NumberFormatException e) {
                Console.printerrln
                    ("invalid port specification " + httpMonitorServer.substring(index + 1));
            }
            
            httpMonitorServer = httpMonitorServer.substring(0, index);
        }
        else if ("local".equals(httpMonitorServer)) {
            httpMonitorServer = null;
        }
        else {
            httpMonitorPort = 80;
        }
        
        if (httpMonitorServer != null) {
            Console.println("sending log data to " + httpMonitorServer + ":" + httpMonitorPort);
            exchangeHandler =
                new HttpMonitorRemoteExchangeHandler(httpMonitorServer, httpMonitorPort);
        }
        else {
            Console.println("storing logs locally into directory " + logFileBaseDir);
            exchangeHandler = new HttpMonitorLogManager(logFileBaseDir);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.htmlmonitor.HttpMonitorComponent#init()
     */
    @Override
    public synchronized void init() throws HttpMonitorException {
        if (server != null) {
            throw new IllegalStateException("already initialized.");
        }
        
        exchangeHandler.init();
        
        exchangeListenerManager = new ExchangeListenerManager(exchangeHandler);
        exchangeListenerManager.init();
        
        Servlet servlet =
            new HttpMonitoringProxyServlet(proxiedServer, proxiedPort, exchangeListenerManager);
        server = new HttpMonitorServer(port, servlet);
        server.init();
      
        shutdownHook = new Thread
            (new ShutdownHook(server, exchangeListenerManager, exchangeHandler));
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.htmlmonitor.HttpMonitorComponent#start()
     */
    @Override
    public synchronized void start() {
        if ((exchangeHandler == null) || (exchangeListenerManager == null) || (server == null)) {
            throw new IllegalStateException("not initialized.");
        }
        
        try {
            Runtime.getRuntime().addShutdownHook(shutdownHook);
            exchangeHandler.start();
            exchangeListenerManager.start();
            server.start();
        }
        catch (HttpMonitorException e) {
            Console.printerrln("could not start HTTP monitoring proxy: " + e);
            Console.logException(e);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.htmlmonitor.HttpMonitorComponent#stop()
     */
    @Override
    public synchronized void stop() {
        if ((exchangeHandler == null) || (exchangeListenerManager == null) || (server == null)) {
            throw new IllegalStateException("not initialized.");
        }
        
        Runtime.getRuntime().removeShutdownHook(shutdownHook);
        server.stop();
        exchangeListenerManager.stop();
        exchangeHandler.stop();
        
        server = null;
        exchangeListenerManager = null;
        exchangeHandler = null;
    }

}
