//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.http;

import java.io.File;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.stream.StreamSource;

import org.xml.sax.SAXException;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.httpmonitor.exchange.HttpExchange;
import de.ugoe.cs.autoquest.httpmonitor.exchange.Session;
import de.ugoe.cs.autoquest.plugin.http.eventcore.HTTPEventType;
import de.ugoe.cs.autoquest.plugin.http.eventcore.HTTPTarget;

/**
 * <p>
 * Parser for HTTP Monitor logs. Uses JAXB for parsing and is therefore quite simple. For each
 * exchange in the log, it creates an appropriate event.
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTTPLogParser {

    /**
     * <p>
     * the event sequences parsed by this parser
     * </p>
     */
    private Collection<List<Event>> sequences = new LinkedList<List<Event>>();

    /**
     * <p>
     * Parses a log file written by the HTTPMonitor and creates a collection of event sequences.
     * </p>
     * 
     * @param filename
     *            name and path of the log file
     *
     * @throws SAXException in the case, the file could not be parsed
     */
    public void parseFile(String filename) throws JAXBException {
        if (filename == null) {
            throw new IllegalArgumentException("filename must not be null");
        }

        parseFile(new File(filename));
    }

    /**
     * <p>
     * Parses a log file written by the HTTPMonitor and creates a collection of event sequences.
     * </p>
     * 
     * @param file
     *            file to be parsed
     *
     * @throws SAXException in the case, the file could not be parsed
     */
    public void parseFile(File file) throws JAXBException {
        if (file == null) {
            throw new IllegalArgumentException("file must not be null");
        }
        
        JAXBContext jc = JAXBContext.newInstance(Session.class.getPackage().getName());
        
        Unmarshaller unmarshaller = jc.createUnmarshaller();
        StreamSource source = new StreamSource(file);
        
        @SuppressWarnings("unchecked")
        JAXBElement<Session> sessionObj = (JAXBElement<Session>) unmarshaller.unmarshal(source);
        
        Session session = sessionObj.getValue();

        if ((session.getHttpExchange() != null) && (session.getHttpExchange().size() > 0)) {
            List<Event> sequence = new LinkedList<Event>();
            for (HttpExchange exchange : session.getHttpExchange()) {
                sequence.add(new Event(new HTTPEventType(exchange),
                                       new HTTPTarget(exchange.getReceiver())));
            }
            sequences.add(sequence);
        }
        
    }

    /**
     * <p>
     * returns the sequences parsed by this parser
     * </p>
     * 
     * @return as described
     */
    public Collection<List<Event>> getSequences() {
        return sequences;
    }
}
