//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.ui.swt;

import java.util.List;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.TreeItem;

import de.ugoe.cs.autoquest.assertions.TextEqualsAssertEventType;
import de.ugoe.cs.autoquest.assertions.TextEqualsReplay;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.IEventTarget;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;

import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;

public class InsertTextEquals extends AbstractInsertEventComposite {
    private Text expectedText;
    private Tree guiTree;

    /**
     * Create the composite.
     * 
     * @param parent
     * @param style
     */
    public InsertTextEquals(Composite parent, int style, GUIModel guiModel) {
        super(parent, style, guiModel);
        setLayout(new GridLayout(3, false));

        Label lblExpectedValue = new Label(this, SWT.NONE);
        lblExpectedValue.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
        lblExpectedValue.setText("Expected Value:");

        expectedText = new Text(this, SWT.BORDER);
        expectedText.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));

        guiTree = new Tree(this, SWT.BORDER);
        guiTree.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 2, 1));
        buildGuiTree();
        new Label(this, SWT.NONE);

        Button btnExpandAll = new Button(this, SWT.NONE);
        btnExpandAll.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                expandAll(guiTree, true);
            }
        });
        btnExpandAll.setText("Expand all");

        Button btnCollapseAll = new Button(this, SWT.NONE);
        btnCollapseAll.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                expandAll(guiTree, false);
            }
        });
        btnCollapseAll.setText("Collapse all");

    }

    @Override
    protected void checkSubclass() {
        // Disable the check that prevents subclassing of SWT components
    }

    @Override
    public Event getEvent() {
        // TODO possibly display error if no target is selected
        TreeItem[] selection = guiTree.getSelection();
        IEventTarget target = null;
        TextEqualsReplay replay = null;
        if (selection.length == 1) {
            target = (IEventTarget) selection[0].getData();
            replay = new TextEqualsReplay(expectedText.getText(), target.toString());
        }

        Event event = new Event(new TextEqualsAssertEventType(), target);

        event.setTarget(target);
        if (replay != null) {
            event.addReplayable(replay);
        }

        return event;
    }

    private void buildGuiTree() {
        for (IGUIElement element : guiModel.getRootElements()) {
            TreeItem child = new TreeItem(guiTree, SWT.NULL);
            child.setText(element.toString());
            child.setData(element);
            buildGuiTree(child, guiModel.getChildren(element));
        }
    }

    private void buildGuiTree(TreeItem currentParent, List<IGUIElement> elements) {
        for (IGUIElement element : elements) {
            TreeItem child = new TreeItem(currentParent, SWT.NULL);
            child.setText(element.toString());
            child.setData(element);
            buildGuiTree(child, guiModel.getChildren(element));
        }
    }

    private void expandAll(Tree tree, boolean expanded) {
        for (TreeItem item : tree.getItems()) {
            expandAll(item, expanded);
        }
    }

    private void expandAll(TreeItem item, boolean expanded) {
        item.setExpanded(expanded);
        for (TreeItem childItem : item.getItems()) {
            expandAll(childItem, expanded);
        }
    }

}
