//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.php.commands;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.php.WeblogParser;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to load sessions from a web log.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class CMDloadWebSequences implements Command {

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#run(java.util.List)
	 */
	@Override
	public void run(List<Object> parameters) {
		if (parameters.size() < 1) {
			throw new IllegalArgumentException();
		}
		String source;
		String sequencesName;
		String serverUrl = null;
		int timeout = -1;
		int minLength = -1;
		int maxLength = -1;
		boolean generateFrequentUsers = false;
		int frequentUserThreshold = 20;
		try {
			source = (String) parameters.get(0);
			sequencesName = (String) parameters.get(1);
			if (parameters.size() >= 3) {
				serverUrl = (String) parameters.get(2);
			}
			if (parameters.size() >= 6) {
				timeout = Integer.parseInt((String) parameters.get(3));
				minLength = Integer.parseInt((String) parameters.get(4));
				maxLength = Integer.parseInt((String) parameters.get(5));
			}
			if (parameters.size() >= 8) {
				generateFrequentUsers = Boolean
						.parseBoolean((String) parameters.get(6));
				frequentUserThreshold = Integer.parseInt((String) parameters
						.get(7));
			}
		} catch (Exception e) {
			throw new IllegalArgumentException();
		}

		WeblogParser parser = new WeblogParser();
		if (serverUrl != null) {
			parser.setUrl(serverUrl);
		}
		if (timeout != -1) {
			parser.setTimeout(timeout);
			parser.setMinLength(minLength);
			parser.setMaxLength(maxLength);
		}
		if (generateFrequentUsers) {
			parser.setFrequentUserThreshold(frequentUserThreshold);
		}
		try {
			parser.parseFile(source);
		} catch (FileNotFoundException e) {
			Console.printerrln(e.getMessage());
		} catch (IOException e) {
			Console.printerrln(e.getMessage());
		} catch (ParseException e) {
			Console.printerrln("Invalid format of date stamps.");
			Console.printerrln(e.getMessage());
		}

		if (GlobalDataContainer.getInstance().addData(sequencesName,
				parser.getSequences())) {
			CommandHelpers.dataOverwritten(sequencesName);
		}
		if (generateFrequentUsers) {
			List<String> frequentUserIDs = parser.getFrequentUsers();
			List<Collection<List<Event>>> frequentUserSessions = parser
					.getFrequentUserSequences();
			for (int i = 0; i < frequentUserIDs.size(); i++) {
				String seqName = sequencesName + "_" + frequentUserIDs.get(i);
				if (GlobalDataContainer.getInstance().addData(seqName,
						frequentUserSessions.get(i))) {
					CommandHelpers.dataOverwritten(seqName);
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#help()
	 */
	@Override
	public String help() {
		return "loadWebSequences <filename> <sequencesName> {<serverUrl>} {<timeout> <minSessionLength> <maxSessionLength>} {<generateFrequentUsers> <frequentUserThreshold>}";
	}

}
