//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.util;

import java.util.List;

import com.google.common.base.CharMatcher;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.collect.HashMultiset;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Multiset;

import de.ugoe.cs.autoquest.eventcore.gui.TextInput;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;

/**
 * <p>
 * Util class to handle text input events/tasks.
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputUtil {

    private TextInputUtil() {
        // util class
    }

    /**
     * 
     * <p>
     * Returns all entered words and signs of all instances of text input events.
     * </p>
     * 
     * @param tasksWithTextInputEvents
     *            tasks with event type {@link TextInput}
     * @return set of all entered word and signs with unique entries
     */
    public static Multiset<String> aggregateEnteredTextFromTextInputs(List<ITask> tasksWithTextInputEvents)
    {
        List<Iterable<String>> allTextInputs = Lists.newArrayList();
        for (ITask taskWithTextInput : tasksWithTextInputEvents) {
            System.out.print("+");
            for (ITaskInstance instance : taskWithTextInput.getInstances()) {
                System.out.print(".");
                TextInput textInput =
                    (TextInput) ((IEventTaskInstance) instance).getEvent().getType();
                allTextInputs.add(splitTextIntoWordsAndSigns(textInput.getEnteredText()));
            }
            System.out.println("");
        }
        System.out.println(allTextInputs);
        return HashMultiset.create(Iterables.concat(allTextInputs));
    }

    /**
     * 
     * <p>
     * Splits entered text into words and signs.
     * </p>
     * 
     * @param enteredText
     *            entered text (e.g. from text input event)
     * @return collection of words and signs
     */
    public static Iterable<String> splitTextIntoWordsAndSigns(String enteredText) {
        CharMatcher onlyWords =
            CharMatcher.WHITESPACE.or(CharMatcher.forPredicate(characterIsNoJavaIdentifierPart()));
        CharMatcher onlySigns =
            CharMatcher.WHITESPACE.or(CharMatcher.forPredicate(characterIsNoJavaIdentifierPart())
                .negate());
        Iterable<String> words =
            Splitter.on(onlyWords).omitEmptyStrings().trimResults().split(enteredText);
        Iterable<String> signs =
            Splitter.on(onlySigns).omitEmptyStrings().trimResults().split(enteredText);
        return Iterables.concat(words, signs);
    }

    /**
     * 
     * <p>
     * Determines, if a character is not part of a Java identifier.
     * </p>
     * 
     * @return true, iff no part of Java identifier
     */
    public static Predicate<Character> characterIsNoJavaIdentifierPart() {
        return new Predicate<Character>() {

            @Override
            public boolean apply(Character character) {
                return !Character.isJavaIdentifierPart(character);
            }

        };
    }

    /**
     * 
     * <p>
     * Determines if the specified character is not a letter or digit.
     * </p>
     * 
     * @return
     */
    public static Predicate<Character> characterIsNoLetterOrDigitPredicate() {
        return new Predicate<Character>() {

            @Override
            public boolean apply(Character character) {
                return !Character.isLetterOrDigit(character);
            }

        };
    }

}
