//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.taskmodel.filter.types;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;

import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.gui.IInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.KeyPressed;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.MouseClick;
import de.ugoe.cs.autoquest.eventcore.gui.MouseInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.Scroll;
import de.ugoe.cs.autoquest.eventcore.gui.TextInput;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;

/**
 * <p>
 * Event type filter for {@link EventTask}s.
 * </p>
 * 
 * @author Alexander Deicke
 */
public enum EventTypeFilter implements TaskFilter<IEventType> {

	KEY_PRESSED(KeyPressed.class),
	
    MOUSE_BUTTON_INTERACTION(MouseButtonInteraction.class),

    MOUSE_CLICK(MouseClick.class),

    MOUSE_INTERACTION(MouseInteraction.class),

    TEXT_INPUT(TextInput.class),

    SCROLL(Scroll.class),

    USER_INTERACTION(IInteraction.class);

    private Class<? extends IEventType> eventTypeClazz;

    private EventTypeFilter(Class<? extends IEventType> eventTypeClazz) {
        this.eventTypeClazz = eventTypeClazz;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usability.tasktree.filters.TaskFilter#getId()
     */
    @SuppressWarnings("unchecked")
    @Override
    public Class<IEventType> clazz() {
        return (Class<IEventType>) eventTypeClazz;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usability.tasktree.filters.TaskFilter#getId()
     */
    @SuppressWarnings("rawtypes")
    @Override
    public Predicate filterPredicate() {
        Predicate<Object> instanceOfIEventTaskPredicate = Predicates.instanceOf(IEventTask.class);
        Predicate<ITask> taskHoldsInstanceOfFilterArgument =
            Predicates.compose(Predicates.instanceOf(eventTypeClazz), taskExtractionFunction());
        return Predicates.and(instanceOfIEventTaskPredicate, taskHoldsInstanceOfFilterArgument);
    }

    /**
     * 
     * <p>
     * Gets the event type of a {@link ITask}.
     * </p>
     * 
     * @return event type
     */
    private Function<ITask, IEventType> taskExtractionFunction() {
        return new Function<ITask, IEventType>() {

            @Override
            public IEventType apply(ITask task) {
                // XXX: Use the type of the first instance provided
                ITaskInstance firstInstance = task.getInstances().iterator().next();
                return ((IEventTaskInstance) firstInstance).getEvent().getType();
            }
        };
    }
}
