//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.usability2.rules.metrics;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.usability.EvaluationMethodCaller;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescriptionResolver;
import de.ugoe.cs.autoquest.usability.rules.UsabilityMetric;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRule;
import de.ugoe.cs.autoquest.plugin.usability2.rules.metrics.visitor.AbstractMetricVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * <p>
 * Metric, which measures the total time spend on running each task by each user. >>>>
 * </p>
 * 
 * @author Konni Hartmann
 */
public class OnTaskTimeMetric extends UsabilityRule implements UsabilityMetric {

    private static final int MINIMUM_NO_OF_INSTANCES = 40;

    /**
     * <p>
     * Constructor. Creates a new {@code OnTaskTimeMetric} for a given task model.
     * </p>
     * 
     * @param taskTree
     */
    public OnTaskTimeMetric(ITaskModel taskModel) {
        super(taskModel);
        this.name = "OnTaskTimeMetric";
        this.defect =
            new UsabilityProblemDescriptionResolver().descriptionFor(this.getClass()
                .getSimpleName());
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usability.rules.UsabilityRule#check()
     */
    @Override
    public Optional<UsabilityProblemDescription> calculate() {
        Collection<ITask> ses = taskModel.getTasks();
        float evaluationMetric = calculateEvaluationMetric(ses);
        return this.defect.isPresent(evaluationMetric);
    }

    static class DataEntry {
        long minTimestamp = Long.MAX_VALUE;
        long maxTimestamp = Long.MIN_VALUE;

        void update(long timestamp) {
            if (timestamp > maxTimestamp)
                maxTimestamp = timestamp;
            if (timestamp < minTimestamp)
                minTimestamp = timestamp;
        }

        long getTimeDifference() {
            return maxTimestamp - minTimestamp;
        }
    }

    class Data {
        Map<Integer, DataEntry> map = new HashMap<Integer, DataEntry>();

        int currentUser = 0;

        void update(Event event) {
            DataEntry data = map.get(currentUser);
            if (data == null) {
                data = new DataEntry();
            }
            data.update(event.getTimestamp());
            map.put(currentUser, data);
        }

        public void setUserID(int currentUser) {
            this.currentUser = currentUser;
        }
    }

    private float calculateEvaluationMetric(Collection<ITask> tasks) {

        ITask maxTask = null;
        DescriptiveStatistics maxVariance = null;
        float result = 0;

        for (ITask task : tasks) {
            Collection<ITaskInstance> instances = task.getInstances();

            final Data data = new Data();
            int userCounter = 0;

            for (ITaskInstance iTaskInstance : instances) {
                AbstractMetricVisitor visitor = new AbstractMetricVisitor() {
                    @Override
                    public void visit(IEventTaskInstance instance) {
                        data.update(instance.getEvent());
                    }
                };

                visitor.visit(iTaskInstance);
                userCounter++;
                data.setUserID(userCounter);
            }

            DescriptiveStatistics stats = new DescriptiveStatistics();
            for (DataEntry entry : data.map.values()) {
                long difference = entry.getTimeDifference();
                if (difference > 0)
                    // ignore single event tasks
                    stats.addValue(difference);
            }
            
            if (stats.getN() == 0)
                continue;
            
            if ((maxVariance == null) ||
                ((stats.getN() > MINIMUM_NO_OF_INSTANCES) && (maxVariance.getStandardDeviation() < stats.getStandardDeviation())))
            {
                maxTask = task;
                maxVariance = stats;
            }
        }

        System.out.println();
        if (maxVariance != null) {
            result = (float) maxVariance.getVariance();
            System.out.printf("(OTT) Highest task: %s (%s)\n", maxTask.getDescription(),
                              maxVariance.getVariance());
            System.out.println(maxVariance.toString());
            System.out.println("Values:");
            for (double val : maxVariance.getValues()) {
                System.out.println(val);
            }
        }
        else
            System.out.println("(OTT) No target found");

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.autoquest.usability.rules.UsabilityRule#callEvaluationMetho(de.ugoe.cs.autoquest
     * .usability.EvaluationMethodCaller)
     */
    @Override
    public Optional<UsabilityProblemDescription> callEvaluationMethod(EvaluationMethodCaller evaluationMethodCaller)
    {
        return evaluationMethodCaller.check(this);
    }
}
