//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeifc;

import java.text.DecimalFormat;

/**
 * <p>
 * represents different metrics available or calculatable for tasks. Measure for the metrics
 * are calculated by a task model and added to the task infos of a specific task.
 * </p>
 * 
 * @author Patrick Harms
 */
public enum TaskMetric {

    COUNT("count", "number of all occurrences of the task in the model"),
    DEPTH("depth",
          "the maximum depth of the task, i.e., the number of children levels including the " +
          "level of the task itself"),
    EVENT_COVERAGE("covered events", "number of all event task instances covered by the task"),
    EVENT_COVERAGE_RATIO("event coverage ratio",
                         "the ratio of events covered by this task in relation to all events " +
                         "covered by all tasks in their instances in per mille", 0.1, "%");
    
    /**
     * <p>
     * the name of the metric
     * </p>
     */
    private String name;
    
    /**
     * <p>
     * a human readable description of the metric
     * </p>
     */
    private String description;

    /**
     * <p>
     * a scale applied for the metric when formatting the value
     * </p>
     */
    private double formatScale;
    
    /**
     * <p>
     * the unit of the metric used when formatting the value
     * </p>
     */
    private String formatUnit;
    
    /**
     * <p>
     * initializes the metric with a name and a description
     * </p>
     */
    private TaskMetric(String name, String description) {
        this.name = name;
        this.description = description;
        this.formatScale = 1.0;
        this.formatUnit = null;
    }

    /**
     * <p>
     * initializes the metric with a name and a description, as well as with a scale and a unit for
     * formatting it.
     * </p>
     */
    private TaskMetric(String name, String description, double formatScale, String formatUnit) {
        this.name = name;
        this.description = description;
        this.formatScale = formatScale;
        this.formatUnit = formatUnit;
    }

    /**
     * <p>
     * returns the name of the metric
     * </p>
     * 
     * @return the name of the metric
     */
    public String getName() {
        return name;
    }

    /**
     * <p>
     * returns the human readable description of the metric
     * </p>
     * 
     * @return the human readable description of the metric
     */
    public String getDescription() {
        return description;
    }
    
    /**
     * <p>
     * formats the provided value of a measure of the metric using the internal format scale and
     * unit.
     * </p>
     * 
     * @return the formatted value depending on the scale and unit of the metric
     */
    public String formatValue(int value) {
        String formattedValue;
        
        if (formatScale != 1.0) {
            double effectiveValue = formatScale * value;
            formattedValue = new DecimalFormat( "#,##0.0;(#)").format(effectiveValue);
        }
        else {
            formattedValue = Integer.toString(value);
        }
        
        if (formatUnit != null) {
            formattedValue += formatUnit;
        }
        
        return formattedValue;
    }
}
