//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.ugoe.cs.autoquest.eventcore.gui.Scroll;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInfo;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.tasktrees.treeifc.TaskMetric;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: 16.07.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class TaskCooccurrenceRule implements UsabilityEvaluationRule {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.usability.UsabilityEvaluationRule#evaluate(TaskTree)
     */
    @Override
    public UsabilityEvaluationResult evaluate(ITaskModel taskModel) {
        UsabilityEvaluationResult results = new UsabilityEvaluationResult(taskModel);

        checkForCooccurrences(results, taskModel);

        return results;
    }

    /**
     * 
     */
    private void checkForCooccurrences(UsabilityEvaluationResult results, ITaskModel taskModel) {
        for (ITask task : taskModel.getTasks()) {
            // only sequences are important for task cooccurrences
            if (task instanceof ISequence) {
                ISequence sequence = (ISequence) task;
                int index1 = 0;
                int index2 = 1;
                List<ITask> children = sequence.getChildren();
                
                while (index2 < children.size()) {
                    ITask task1 = children.get(index1);
                    ITask task2 = children.get(index2);
                    ITaskInfo info1 = taskModel.getTaskInfo(task1);
                    ITaskInfo info2 = taskModel.getTaskInfo(task2);
                    
                    int ratioTask1 = 1000 * info1.getMeasureValue(TaskMetric.COUNT, sequence) /
                        info1.getMeasureValue(TaskMetric.COUNT);
                    int ratioTask2 = 1000 * info2.getMeasureValue(TaskMetric.COUNT, sequence) /
                            info2.getMeasureValue(TaskMetric.COUNT);
                    
                    createSucceededDefectIfRequired(ratioTask1, task1, task2, results, taskModel);
                    createPrecededDefectIfRequired(ratioTask2, task1, task2, results, taskModel);
                    
                    index1 = index2;
                    index2++;
                }
            }
        }
    }

    /**
     *
     */
    private void createSucceededDefectIfRequired(int                       ratio,
                                                 ITask                     task1,
                                                 ITask                     task2,
                                                 UsabilityEvaluationResult results,
                                                 ITaskModel                taskModel)
    {
        //TODO document magic numbers
        UsabilityDefectSeverity severity = UsabilityDefectSeverity.getSeverity
            (ratio, 900, 700, 500, 300, task1, taskModel);

        if (!isScroll(task1) && !isScroll(task2) && (severity != null)) {
            Map<String, Object> parameters = new HashMap<String, Object>();
            parameters.put("task1", task1);
            parameters.put("task2", task2);
            parameters.put("ratio", (ratio / 10));

            results.addDefect(severity, UsabilityDefectDescription.COOCCURENCE_SUCCEED, parameters);
        }
    }

    /**
     *
     */
    private void createPrecededDefectIfRequired(int                       ratio,
                                                ITask                     task1,
                                                ITask                     task2,
                                                UsabilityEvaluationResult results,
                                                ITaskModel                taskModel)
    {
        //TODO document magic numbers
        UsabilityDefectSeverity severity = UsabilityDefectSeverity.getSeverity
            (ratio, 900, 700, 500, 300, task2, taskModel);

        if (!isScroll(task1) && !isScroll(task2) && (severity != null)) {
            Map<String, Object> parameters = new HashMap<String, Object>();
            parameters.put("task1", task1);
            parameters.put("task2", task2);
            parameters.put("ratio", (ratio / 10));

            results.addDefect(severity, UsabilityDefectDescription.COOCCURENCE_PRECED, parameters);
        }
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param task1
     * @return
     */
    private boolean isScroll(ITask task) {
        if (task instanceof IEventTask) {
            return ((IEventTaskInstance) ((IEventTask) task).getInstances().iterator().next()).getEvent().getType() instanceof Scroll;
        }
        else if (task instanceof IIteration) {
            return isScroll(((IIteration) task).getMarkedTask());
        }
        
        return false;
    }
}
