//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import static de.ugoe.cs.autoquest.usability.UsabilityDefectDescription.COOCCURENCE_PRECED;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectDescription.COOCCURENCE_SUCCEED;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.HIGH;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.INFO;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.LOW;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.MEDIUM;

import org.junit.Before;
import org.junit.Test;

import de.ugoe.cs.autoquest.usability.UsabilityDefect;

/**
 *
 */
public class TaskCooccurrenceRuleTest extends AbstractUsabilityEvaluationTC {

    /**
     *
     */
    @Before
    public void setUp() {
        UsabilityDefectSeverity.defaultCoverageQuantile = 0;
    }

    /**
     *
     */
    @Test
    public void testCooccurrence_01() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Interaction elem1 {}" +
            "  Interaction elem2 {}" +
            "}";
        
        // no defect expected, as interactions do not form tasks
        UsabilityDefect[] expectedDefects = new UsabilityDefect[] {  };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testCooccurrence_02() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Interaction elem1 {}" +
            "  Interaction elem2 {}" +
            "  Interaction elem1 {}" +
            "  Interaction elem2 {}" +
            "}";
        
        // no defect expected, as interactions do not form tasks
        UsabilityDefect[] expectedDefects = new UsabilityDefect[] { };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testCooccurrence_03() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testCooccurrence_04() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_01() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "    Interaction elem3 {}" +
            "    Interaction elem4 {}" +
            "    Interaction elem5 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_02() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(LOW, COOCCURENCE_PRECED),
              new UsabilityDefect(INFO, COOCCURENCE_PRECED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_03() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem2 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem2 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(LOW, COOCCURENCE_SUCCEED),
              new UsabilityDefect(INFO, COOCCURENCE_SUCCEED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_04() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
           { new UsabilityDefect(LOW, COOCCURENCE_PRECED),
             new UsabilityDefect(LOW, COOCCURENCE_PRECED),
             new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
             new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_05() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem2 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem2 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
           { new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
             new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
             new UsabilityDefect(LOW, COOCCURENCE_SUCCEED),
             new UsabilityDefect(LOW, COOCCURENCE_SUCCEED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_06() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(MEDIUM, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_07() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem2 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(MEDIUM, COOCCURENCE_SUCCEED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_08() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction other {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(MEDIUM, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED),
              new UsabilityDefect(HIGH, COOCCURENCE_SUCCEED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testSeldomCooccurrence_09() {
        TaskCooccurrenceRule rule = new TaskCooccurrenceRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Interaction elem2 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction other {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(HIGH, COOCCURENCE_PRECED),
              new UsabilityDefect(MEDIUM, COOCCURENCE_SUCCEED) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }
}
