//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.guimodel;

import de.ugoe.cs.autoquest.eventcore.guimodel.IDialog;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;

/**
 * <p>
 * a GUI element representing an HTML document, i.e., a page on an HTML server. This is the
 * element for a GUI model of an HTML web site being always and only the children of servers. It
 * is identified through the server on which it resists, its path, a potential query, and a title.
 * Its children are HTML page elements
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTMLDocument extends HTMLGUIElement implements IDialog {

    /**
     * <p>
     * default serial version UID
     * </p> 
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * instantiates the document with it specification and its parent, which is always a server
     * </p>
     *
     * @param specification the specification of the document
     * @param parent        the server on which the document resists
     */
    public HTMLDocument(HTMLDocumentSpec specification, HTMLServer parent) {
        super(specification, parent);
    }

    /**
     * <p>
     * returns the path in the URL of the document
     * </p>
     *
     * @return the path in the URL of the document
     */
    public String getPath() {
        return ((HTMLDocumentSpec) super.getSpecification()).getPath();
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.html.guimodel.HTMLGUIElement#getElementDescriptor()
     */
    @Override
    protected String getElementDescriptor() {
        return "Document";
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#getDistanceTo(IGUIElement)
     */
    @Override
    public double getDistanceTo(IGUIElement otherElement) {
        if (otherElement instanceof HTMLServer) {
            // use the implementation provided by the server
            return otherElement.getDistanceTo(this);
        }
        else if (otherElement instanceof HTMLDocument) {
            if (this.getSpecification().getSimilarity(otherElement.getSpecification())) {
                return DISTANCE_NONE;
            }
            else {
                // use the implementation provided by the server to check if the document resists
                // at least on the same server
                return getServer().getDistanceTo(otherElement);
            }
        }
        else if (otherElement instanceof HTMLPageElement) {
            HTMLDocumentSpec documentSpec =
                ((HTMLPageElementSpec) otherElement.getSpecification()).getPage();
            
            if (this.getSpecification().getSimilarity(documentSpec)) {
                return DISTANCE_SAME_DOCUMENT;
            }
            else {
                // use the implementation provided by the server to check if the document resists
                // at least on the same server
                return getServer().getDistanceTo(otherElement);
            }
        }
        
        return DISTANCE_DISTINCT_SERVER;
    }

    /**
     * <p>
     * returns the server on which the document resists
     * </p>
     *
     * @return the server on which the document resists
     */
    HTMLServer getServer() {
        return (HTMLServer) super.getParent();
    }

    /**
     * <p>
     * returns the title of the document
     * </p>
     *
     * @return the title of the document
     */
    String getTitle() {
        return ((HTMLDocumentSpec) super.getSpecification()).getTitle();
    }

}
