//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.http.commands;

import java.io.File;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.logging.Level;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.http.HTTPLogParser;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command that tries to parse all files in a folder as if they were log files generated by the
 * HTTPMonitor. The result is one set of sequences for all files (not one set of sequences for each
 * file!).
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDparseDirHTTP implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String directory = null;
        String sequencesName = "sequences";

        try {
            directory = (String) parameters.get(0);
            if (parameters.size() >= 2) {
                sequencesName = (String) parameters.get(1);
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("missing parameter for directory");
        }

        File folder = new File(directory);
        if (!folder.isDirectory()) {
            Console.printerrln(directory + " is not a directory");
            return;
        }

        HTTPLogParser parser = new HTTPLogParser();

        parseFile(folder, parser);

        Collection<List<Event>> sequences = parser.getSequences();

        if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
            CommandHelpers.dataOverwritten(sequencesName);
        }
    }

    /**
     * <p>
     * recursive method for parsing a directory structures
     * </p>
     *
     * @param file   the file object to be parsed. If the file is a folder, the method calls itself
     *               for all children
     * @param parser the parser to use for parsing the files.
     */
    private void parseFile(File file, HTTPLogParser parser) {
        if (file.isDirectory()) {
            String[] children = file.list();
            Arrays.sort(children);
            
            for (String child : children) {
                File childFile = new File(file, child);
                parseFile(childFile, parser);
            }
        }
        else if (file.isFile()) {
            String source = file.getAbsolutePath();
            Console.traceln(Level.INFO, "Processing file: " + source);

            try {
                parser.parseFile(file);
            }
            catch (Exception e) {
                Console.printerrln("Could not parse " + source + ": " + e.getMessage());
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "parseDirHTTP <directory> [<sequencesName>]";
    }

}
