//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.gui.Scroll;
import de.ugoe.cs.autoquest.tasktrees.treeifc.DefaultTaskInstanceTraversingVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IMarkingTemporalRelationship;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: 16.07.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class RequiredInefficientActionsRule implements UsabilityEvaluationRule {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.usability.UsabilityEvaluationRule#evaluate(TaskTree)
     */
    @Override
    public UsabilityEvaluationResult evaluate(ITaskModel taskModel) {
        UsabilityEvaluationResult results = new UsabilityEvaluationResult(taskModel);

        Map<ITask, double[]> smellingTasks = getInefficientActionStatistics(taskModel.getTasks());
        analyzeTasksWithInefficientActions(smellingTasks, results, taskModel);

        return results;
    }

    /**
     *
     */
    private void analyzeTasksWithInefficientActions(Map<ITask, double[]>      smellingTasks,
                                                    UsabilityEvaluationResult results,
                                                    ITaskModel                taskModel)
    {

        for (Map.Entry<ITask, double[]> entry : smellingTasks.entrySet()) {
            DescriptiveStatistics stats = new DescriptiveStatistics(entry.getValue());
            
            int ratio = (int) (1000 * stats.getMean());

            UsabilityDefectSeverity severity = UsabilityDefectSeverity.getSeverity
                (ratio, 500, 300, 200, 100, entry.getKey(), taskModel);

            if (severity != null) {
                Map<String, Object> parameters = new HashMap<String, Object>();
                parameters.put("task", entry.getKey());
                parameters.put("ratio", (ratio / 10));

                results.addDefect
                    (severity, UsabilityDefectDescription.INEFFICIENT_ACTIONS, parameters);
            }
        }
    }

    /**
     * 
     */
    private Map<ITask, double[]> getInefficientActionStatistics(Collection<ITask> tasks) {
        Map<ITask, double[]> inefficientActionRatios = new HashMap<ITask, double[]>();
        
        for (ITask task : tasks) {
            if (taskMustBeChecked(task))  {
                double[] ratios = getRatiosOfInefficientActionsInInstances(task);
                
                for (int i = 0; i < ratios.length; i++) {
                    if (ratios[i] > 0) {
                        // there is at least on inefficient action
                        inefficientActionRatios.put(task, ratios);
                        break;
                    }
                }
            }
        }
        
        return inefficientActionRatios;
    }

    /**
     *
     */
    private boolean taskMustBeChecked(ITask task) {
        if ((task instanceof IEventTask) || (task instanceof IMarkingTemporalRelationship)) {
            // event tasks are not considered
            // marking temporal relationships have a child, that is more important, but it will
            // be checked independently as all tasks of a task model are checked separately
            return false;
        }
        else {
            return true;
        }
    }

    /**
     *
     */
    private double[] getRatiosOfInefficientActionsInInstances(ITask task) {
        Collection<ITaskInstance> instances = task.getInstances();
        
        double[] ratios = new double[instances.size()];
        int index = 0;
        for (ITaskInstance instance : instances) {
            ratios[index++] = getRatioOfInefficientActionsInInstance(instance);
        }
        
        return ratios;
    }

    /**
     *
     */
    private double getRatioOfInefficientActionsInInstance(ITaskInstance instance) {
        final int[] count = new int[2];
        count[0] = 0;
        count[1] = 0;
        
        instance.accept(new DefaultTaskInstanceTraversingVisitor() {
            @Override
            public void visit(IEventTaskInstance eventTaskInstance) {
                if (isInefficientAction(eventTaskInstance.getEvent())) {
                    count[0]++;
                }
                
                count[1]++;
            }
            
        });
        
        return (double) count[0] / count[1];
    }

    /**
     *
     */
    private boolean isInefficientAction(Event event) {
        return (event.getType() instanceof Scroll);
    }

}
