//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.mfc.guimodel;

import de.ugoe.cs.autoquest.eventcore.guimodel.AbstractDefaultGUIElement;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec;

/**
 * <p>
 * Base class that represents GUI element in MFC GUIs.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public abstract class MFCGUIElement extends AbstractDefaultGUIElement {

    /**
     * <p>
     * Id for object serialization.
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * Constructor. Creates a new MFCGUIElement.
     * </p>
     * 
     * @param specification
     *            specification of created GUI element
     * @param parent
     *            parent of the created GUI element; null means that the element is a top-level
     *            window
     */
    public MFCGUIElement(MFCGUIElementSpec specification, MFCGUIElement parent) {
        super(specification, parent);
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.eventcore.IEventTarget#getPlatform()
     */
    @Override
    public String getPlatform() {
        return "MFC";
    }

    /**
     * <p>
     * Returns the HWND (Id) of the GUI element.
     * </p>
     * 
     * @return the HWND (Id)
     */
    public String getId() {
        return Long.toString(((MFCGUIElementSpec) super.getSpecification()).getHwnd());
    }

    /**
     * <p>
     * Returns the type of the GUI element.
     * </p>
     * 
     * @return the type
     */
    public String getType() {
        return ((MFCGUIElementSpec) super.getSpecification()).getType();
    }

    /**
     * <p>
     * Returns the name of the GUI element.
     * </p>
     * 
     * @return the name
     */
    public String getName() {
        return ((MFCGUIElementSpec) super.getSpecification()).getName();
    }

    /**
     * <p>
     * Returns the modality of the GUI element.
     * </p>
     * 
     * @return the modality
     */
    public boolean isModal() {
        return ((MFCGUIElementSpec) super.getSpecification()).isModal();
    }

    /**
     * <p>
     * Returns the resource Id of the GUI element.
     * </p>
     * 
     * @return the resource Id
     */
    public int getResourceId() {
        return ((MFCGUIElementSpec) super.getSpecification()).getResourceId();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#updateSpecification(de.ugoe.cs.autoquest.eventcore
     * .guimodel.IGUIElementSpec)
     */
    @Override
    public void updateSpecification(IGUIElementSpec furtherSpec) {
        ((MFCGUIElementSpec) super.getSpecification()).update(furtherSpec);
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.eventcore.IEventTarget#getStringIdentifier()
     */
    @Override
    public String getStringIdentifier() {
        String str = this.toString();
        if (getParent() != null) {
            return str + "<-" + getParent().getStringIdentifier();
        }
        return str;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return super.getSpecification().toString();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#getDistanceTo(IGUIElement)
     */
    @Override
    public double getDistanceTo(IGUIElement otherElement) {
        throw new UnsupportedOperationException("not implemented yet");
    }

    /**
     * <p>
     * Returns the XML representation of the GUI element.
     * </p>
     * 
     * @return the XML representation
     */
    public String toXML() {
        if (getParent() != null) {
            return ((MFCGUIElement) getParent()).toXML() +
                ((MFCGUIElementSpec) super.getSpecification()).toXML();
        }
        else {
            return ((MFCGUIElementSpec) super.getSpecification()).toXML();
        }
    }
}
