//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.eventcore.gui;

import java.util.LinkedList;
import java.util.List;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;

/**
 * <p>
 * This class iterates the provided sequence and sets the target of all key interaction events
 * to the GUI element having the current keyboard focus. The current keyboard focus is determined
 * either by keyboard focus events or by using the target of the first key interaction in the
 * provided sequence. Events changing the keyboard focus are discarded herewith.
 * </p>
 * 
 * @version $Revision: $ $Date: 29.08.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class KeyInteractionTargetCorrector {
    
    /**
     * <p>
     * This method performs the work described in the description of the class. A new list is
     * instantiated and returned. This list is filled with the events provided by the sequence
     * being the parameter of the method except for key interaction events. Those are replaced
     * by a new event with the identical event type but the corrected event target.
     * </p>
     * 
     * @param sequence the event sequence to correct the key interactions targets in
     * 
     * @return the resulting sequence, in which key interactions have the correct target, i.e.
     *         the GUI element having the keyboard focus
     */
    public List<Event> correctKeyInteractionTargets(List<Event> sequence) {
        List<Event> resultingSequence = new LinkedList<Event>();
        IGUIElement currentKeyboardFocusGUIElement = null;
        
        for (Event event : sequence) {
            Event nextEvent = event;
            if (event.getTarget() instanceof IGUIElement) {
                if (event.getType() instanceof KeyboardFocusChange) {
                    currentKeyboardFocusGUIElement = (IGUIElement) event.getTarget();
                    nextEvent = null;
                }
                else if (event.getType() instanceof KeyInteraction) {
                    if (currentKeyboardFocusGUIElement == null) {
                        currentKeyboardFocusGUIElement = (IGUIElement) event.getTarget();
                    }
                    
                    if (!currentKeyboardFocusGUIElement.equals(event.getTarget())) {
                        nextEvent = new Event(event.getType(), currentKeyboardFocusGUIElement);
                        nextEvent.setTimestamp(event.getTimestamp());
                        
                        // TODO copy event parameters
                    }
                }
            }
            
            if (nextEvent != null) {
                resultingSequence.add(nextEvent);
            }
        }
        
        return resultingSequence;
    }
}
