//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import static de.ugoe.cs.autoquest.usability.UsabilityDefectDescription.INEFFICIENT_ACTIONS;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.HIGH;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.INFO;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.LOW;
import static de.ugoe.cs.autoquest.usability.UsabilityDefectSeverity.MEDIUM;

import org.junit.Before;
import org.junit.Test;

import de.ugoe.cs.autoquest.usability.UsabilityDefect;

/**
 *
 */
public class RequiredInefficientActionsRuleTest extends AbstractUsabilityEvaluationTC {

    /**
     *
     */
    @Before
    public void setUp() {
        UsabilityDefectSeverity.defaultCoverageQuantile = 0;
    }
    
    /**
     *
     */
    @Test
    public void testWithNormalScroll_01() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Scroll body {}" +
            "  Interaction elem1 {}" +
            "}";
        
        // no defect expected, as interactions do not form tasks
        UsabilityDefect[] expectedDefects = new UsabilityDefect[] {  };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_02() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Scroll body {}" +
            "  Interaction elem1 {}" +
            "  Scroll body {}" +
            "  Interaction elem1 {}" +
            "}";
        
        // no defect expected, as interactions do not form tasks
        UsabilityDefect[] expectedDefects = new UsabilityDefect[] { };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_03() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_04() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_05() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_06() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(HIGH, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_01() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem1 {}" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(LOW, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_02() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(MEDIUM, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_03() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(MEDIUM, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_04() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
           { new UsabilityDefect(LOW, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_05() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
           { new UsabilityDefect(LOW, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_06() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(INFO, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_07() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[]
            { new UsabilityDefect(INFO, INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_08() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[] { };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_09() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "}";
        
        UsabilityDefect[] expectedDefects = new UsabilityDefect[] { };

        assertUsabilityEvaluationResult(expectedDefects, rule.evaluate(createTaskModel(spec)));

    }
}
