//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usageprofiles;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>
 * Default implementation of the {@link SymbolMap}. Uses internally a {@link HashMap} for
 * implementing the interface by delegating the calls to this class to the internal map.
 * </p>
 * 
 * @author Patrick Harms
 */
public class DefaultSymbolMap<K, V> implements SymbolMap<K, V> {

    /**
     * default serial version id
     */
    private static final long serialVersionUID = 1L;
    
    /**
     * the internally used map for implementing this interface
     */
    private Map<K, V> delegate;

    /**
     * <p>
     * creates a new symbol map
     * </p>
     */
    public DefaultSymbolMap() {
        delegate = new HashMap<K, V>();
    }

    /**
     * <p>
     * creates a copy of an existing symbol map. The other map must not be null.
     * </p>
     *
     * @param other the map to be copied
     */
    public DefaultSymbolMap(SymbolMap<K, V> other) {
        if (other == null) {
            throw new IllegalArgumentException("other map must not be null");
        }
        
        delegate = new HashMap<K, V>();
        
        for (K symbol : other.getSymbols()) {
            delegate.put(symbol, other.getValue(symbol));
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#size()
     */
    @Override
    public int size() {
        return delegate.size();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#isEmpty()
     */
    @Override
    public boolean isEmpty() {
        return delegate.isEmpty();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#containsSymbol(java.lang.Object)
     */
    @Override
    public boolean containsSymbol(K symbol) {
        if (symbol == null) {
            throw new IllegalArgumentException("symbol must not be null");
        }
        
        return delegate.containsKey(symbol);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#getValue(java.lang.Object)
     */
    @Override
    public V getValue(K symbol) {
        if (symbol == null) {
            throw new IllegalArgumentException("symbol must not be null");
        }
        
        return delegate.get(symbol);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#addSymbol(java.lang.Object, java.lang.Object)
     */
    @Override
    public void addSymbol(K symbol, V value) {
        if (symbol == null) {
            throw new IllegalArgumentException("symbol must not be null");
        }
        
        delegate.put(symbol, value);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#removeSymbol(java.lang.Object)
     */
    @Override
    public V removeSymbol(K symbol) {
        if (symbol == null) {
            throw new IllegalArgumentException("symbol must not be null");
        }
        
        return delegate.remove(symbol);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#getSymbols()
     */
    @Override
    public Collection<K> getSymbols() {
        return delegate.keySet();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#getValues()
     */
    @Override
    public Collection<V> getValues() {
        return delegate.values();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#clear()
     */
    @Override
    public void clear() {
        delegate.clear();
    }

    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return delegate.hashCode();
    }

    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        else if (this.getClass().isInstance(obj)) {
            return delegate.equals(((DefaultSymbolMap<K, V>) obj).delegate);
        }
        else {
            return false;
        }
    }

}
