//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.guimodel;

import de.ugoe.cs.autoquest.eventcore.guimodel.IFrame;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;

/**
 * <p>
 * a GUI element representing an HTML server. This is the root element for a GUI model of
 * an HTML web site. It is identified through a host and a port. Its children are documents.
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTMLServer extends HTMLGUIElement implements IFrame {

    /**
     * <p>
     * default serial version UID
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * instantiates the server with an appropriate specification
     * </p>
     *
     * @param specification the server specification
     * @param parent        the parent of the server --> must always be null. Just included as
     *                      required by the automatic instantiation mechanism
     *                      
     * @throws IllegalArgumentException if the provided parent is not null
     */
    public HTMLServer(HTMLServerSpec specification, HTMLGUIElement parent) {
        super(specification, null);
        
        if (parent != null) {
            throw new IllegalArgumentException("a GUI element representing the server must not " +
                                               "have a parent GUI element");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.html.guimodel.HTMLGUIElement#getElementDescriptor()
     */
    @Override
    protected String getElementDescriptor() {
        return "Server";
    }

    /**
     * <p>
     * returns the host of the represented server
     * </p>
     *
     * @return the host
     */
    public String getHost() {
        return ((HTMLServerSpec) super.getSpecification()).getHost();
    }

    /**
     * <p>
     * returns the port of the represented server
     * </p>
     *
     * @return the port
     */
    public int getPort() {
        return ((HTMLServerSpec) super.getSpecification()).getPort();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#getDistanceTo(IGUIElement)
     */
    @Override
    public double getDistanceTo(IGUIElement otherElement) {
        if (otherElement instanceof HTMLServer) {
            if (this.equals(otherElement)) {
                return DISTANCE_NONE;
            }
            else {
                return DISTANCE_DISTINCT_SERVER;
            }
        }
        else if (otherElement instanceof HTMLDocument) {
            if (this.equals(((HTMLDocument) otherElement).getServer())) {
                return DISTANCE_SAME_SERVER;
            }
            else {
                return DISTANCE_DISTINCT_SERVER;
            }
        }
        else if (otherElement instanceof HTMLPageElement) {
            HTMLServerSpec serverSpec =
                ((HTMLPageElementSpec) otherElement.getSpecification()).getPage().getServer();
            
            if (this.getSpecification().getSimilarity(serverSpec)) {
                return DISTANCE_SAME_SERVER;
            }
            else {
                return DISTANCE_DISTINCT_SERVER;
            }
        }
        
        return DISTANCE_DISTINCT_SERVER;
    }

}
