//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeimpl;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIterationInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IOptionalInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelectionInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequenceInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstanceList;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskVisitor;

/**
 * <p>
 * this is the default implementation of the interface {@link ITask}. It
 * does not do anything fancy except implementing the interface.
 * </p>
 * 
 * @author Patrick Harms
 */
abstract class Task implements ITask {

    /**
     * <p>
     * default serial version UID
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * used as a counter to generate new ids for each newly created task. May overflow.
     * </p>
     */
    private static int temporalId = 0;

    /**
     * <p>
     * the id of the task (unique throughout the system as long as {@link #temporalId} does not
     * overflow.
     * </p>
     */
    private int id;
    
    /**
     * <p>
     * a human readable type of the task (used for visualization purposes)
     * </p>
     */
    private String type;

    /**
     * <p>
     * a human readable description of the task
     * </p>
     */
    private String description;
    
    /**
     * <p>
     * the instances of this task
     * </p>
     */
    private Collection<ITaskInstance> instances = new HashSet<ITaskInstance>();
    
    /**
     * <p>
     * the execution variants of this task
     * </p>
     */
    private Collection<Collection<ITaskInstance>> executionVariants;

    /**
     * <p>
     * constructs a new task with a new id. The id is generated using the {@link #getNewId()}
     * method
     * </p>
     * 
     * @param type the human readable type of the task
     * 
     * @throws IllegalArgumentException in the case the provided type is null
     */
    Task(String type) {
        this.id = getNewId();
        this.type = type;
        
        if (type == null) {
            throw new IllegalArgumentException("type must not be null");
        }
    }

    /**
     * <p>
     * creates a new id for a task using {@link #temporalId} by incrementing it an returning its
     * current value. Resets the counter if {@link Integer.MAX_VALUE} is reached.
     * </p>
     * 
     * @return a new unique id for a task as long as {@link #temporalId} does not overflow
     */
    private static synchronized int getNewId() {
        if (temporalId == Integer.MAX_VALUE) {
            temporalId = 0;
        }

        return temporalId++;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#geId()
     */
    @Override
    public int getId() {
        return id;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#getType()
     */
    @Override
    public String getType() {
        return type;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#getDescription()
     */
    @Override
    public String getDescription() {
        return description;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#getInstances()
     */
    @Override
    public Collection<ITaskInstance> getInstances() {
        return Collections.unmodifiableCollection(instances);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#getExecutionVariants()
     */
    @Override
    public synchronized Collection<Collection<ITaskInstance>> getExecutionVariants() {
        if (executionVariants == null) {
            executionVariants = new LinkedList<Collection<ITaskInstance>>();
            determineExecutionVariants(executionVariants);
        }
        
        return executionVariants;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#equals(ITask)
     */
    @Override
    public final boolean equals(ITask task) {
        // tasks are only equal if they are identical or if they have the same id
        // (may happen, if they are cloned)
        return (this == task) || (this.hashCode() == task.hashCode());
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#hashCode()
     */
    @Override
    public synchronized int hashCode() {
        return id;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public synchronized String toString() {
        StringBuffer result = new StringBuffer();
        result.append(type);
        result.append(" #");
        result.append(id);
        
        if (description != null) {
            result.append(" (");
            result.append(description);
            result.append(')');
        }
        
        return result.toString();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#clone()
     */
    @Override
    public synchronized ITask clone() {
        Task clone = null;
        try {
            clone = (Task) super.clone();
        }
        catch (CloneNotSupportedException e) {
            // this should never happen. Therefore simply dump the exception
            e.printStackTrace();
        }

        return clone;
    }

    /**
     * <p>
     * internally used to set the human readable description of the task
     * </p>
     * 
     * @param description the new human readable description of the task
     */
    void setDescription(String description) {
        this.description = description;
    }

    /**
     * <p>
     * internally used to remove an instance from this task
     * </p>
     * 
     * @param instance the instance to be removed from this task
     */
    synchronized void removeInstance(ITaskInstance instance) {
        this.instances.remove(instance);
        this.executionVariants = null;
    }

    /**
     * <p>
     * internally used to add an instance to this task
     * </p>
     * 
     * @param instance the instance belonging to this task
     */
    synchronized void addInstance(ITaskInstance instance) {
        this.instances.add(instance);
        this.executionVariants = null;
    }
    
    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITask#accept(ITaskVisitor)
     */
    @Override
    public void accept(ITaskVisitor visitor) {
        visitor.visit(this);
    }

    /**
     *
     */
    private void determineExecutionVariants(Collection<Collection<ITaskInstance>> executionVariants)
    {
        for (ITaskInstance instance : instances) {
            boolean added = false;
            for (Collection<ITaskInstance> variant : executionVariants) {
                if (!variant.isEmpty() && (isSameExecution(variant.iterator().next(), instance))) {
                    variant.add(instance);
                    added = true;
                }
            }
            
            if (!added) {
                Collection<ITaskInstance> variant = new HashSet<ITaskInstance>();
                variant.add(instance);
                executionVariants.add(variant);
            }
        }
    }

    /**
     *
     */
    private boolean isSameExecution(ITaskInstance instance1, ITaskInstance instance2) {
        if (instance1 instanceof IIterationInstance) {
            if (!(instance2 instanceof IIterationInstance)) {
                return false;
            }
            
            ITaskInstanceList iteration1 = (ITaskInstanceList) instance1;
            ITaskInstanceList iteration2 = (ITaskInstanceList) instance2;
            
            return isSameExecutionList(iteration1, iteration2);
        }
        else if (instance1 instanceof ISequenceInstance) {
            if (!(instance2 instanceof ISequenceInstance)) {
                return false;
            }
            
            ITaskInstanceList selection1 = (ITaskInstanceList) instance1;
            ITaskInstanceList selection2 = (ITaskInstanceList) instance2;
            
            return isSameExecutionList(selection1, selection2);
        }
        else if (instance1 instanceof ISelectionInstance) {
            if (!(instance2 instanceof ISelectionInstance)) {
                return false;
            }
            else {
                return isSameExecution(((ISelectionInstance) instance1).getChild(),
                                       ((ISelectionInstance) instance2).getChild());
            }
        }
        else if (instance1 instanceof IOptionalInstance) {
            if (!(instance2 instanceof IOptionalInstance)) {
                return false;
            }
            else {
                return isSameExecution(((IOptionalInstance) instance1).getChild(),
                                       ((IOptionalInstance) instance2).getChild());
            }
        }
        else if (instance1 instanceof IEventTaskInstance) {
            if (!(instance2 instanceof IEventTaskInstance)) {
                return false;
            }
            else {
                return ((IEventTaskInstance) instance1).getTask().equals
                    (((IEventTaskInstance) instance2).getTask());
            }
        }
        else if (instance1 == null) {
            return instance2 == null;
        }
        else {
            throw new IllegalArgumentException("unknown type of task instance: " + instance1);
        }
    }

    /**
     *
     */
    private boolean isSameExecutionList(ITaskInstanceList list1, ITaskInstanceList list2) {
        if (list1.size() == list2.size()) {
            for (int i = 0; i < list1.size(); i++) {
                if (!isSameExecution(list1.get(i), list2.get(i))) {
                    return false;
                }
            }
            
            return true;
        }
        else {
            return false;
        }
    }
}
