//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.mfc;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import de.ugoe.cs.autoquest.commands.sequences.CMDgenerateReplayfile;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.plugin.mfc.LogPreprocessor;
import de.ugoe.cs.autoquest.plugin.mfc.MFCLogParser;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;
import de.ugoe.cs.util.console.listener.IOutputListener;

/**
 * @author Steffen Herbold
 */
public class MFCLogParserTest implements IOutputListener {

    /** */
    private File tmpFile;
    
    /** */
    private PrintStream outputStream;

    /**
     * 
     */
    @Before
    public void setUp() throws Exception {
        outputStream = new PrintStream(new FileOutputStream("blub.log"));
        Console.getInstance().registerOutputListener(this);
        
        tmpFile = new File("tmp.xml");
        
        if (tmpFile.exists()) {
            assertTrue(tmpFile.delete());
        }
    }
    
    /**
     * 
     */
    @After
    public void tearDown() throws Exception {
        if (!System.out.equals(outputStream) && !System.err.equals(outputStream)) {
            outputStream.close();
        }
        
        if (tmpFile.exists()) {
            assertTrue(tmpFile.delete());
        }
    }
    
    
    /**
     *
     */
    @Test
    public void testTrace() throws Exception {
        File traceFile = new File(ClassLoader.getSystemResource("trace.txt").getFile());
        
        new LogPreprocessor().convertToXml(traceFile.getAbsolutePath(),
                                           tmpFile.getAbsolutePath());
        
        MFCLogParser parser = new MFCLogParser();
        parser.parseFile(tmpFile);
        Collection<List<Event>> events = parser.getSequences();

        assertNotNull(events);
        assertTrue(events.size() > 0);

        System.err.println("{");
        for (List<Event> session : events) {
            System.err.println("  {");
            for (Event event : session) {
                System.err.print("    ");
                System.err.print(event);
                System.err.println(",");
            }
            System.err.println("  }");
        }
        System.err.println("}");
        System.err.println("\n\n");

        GUIModel guiModel = parser.getGuiModel();
        assertNotNull(guiModel);
        assertNotSame(0, guiModel.getRootElements().size());

        guiModel.dump(System.err, "UTF-8");
    }

    /**
     *
     */
    @Test
    public void testLogfile() throws Exception {
        File logFile = new File(ClassLoader.getSystemResource("log.xml").getFile());

        MFCLogParser parser = new MFCLogParser();
        parser.parseFile(logFile);
        Collection<List<Event>> events = parser.getSequences();

        assertNotNull(events);
        assertTrue(events.size() > 0);

        System.err.println("{");
        for (List<Event> session : events) {
            System.err.println("  {");
            for (Event event : session) {
                System.err.print("    ");
                System.err.print(event);
                System.err.println(",");
            }
            System.err.println("  }");
        }
        System.err.println("}");
        System.err.println("\n\n");

        GUIModel guiModel = parser.getGuiModel();
        assertNotNull(guiModel);

        guiModel.dump(System.err, "UTF-8");

        
        GlobalDataContainer.getInstance().addData("seqs",  parser.getSequences());
        
        List<Object> parameters = new ArrayList<Object>();
        parameters.add(tmpFile.getAbsolutePath());
        parameters.add("seqs");
        
        CMDgenerateReplayfile cmd = new CMDgenerateReplayfile();
        cmd.run(parameters);
        
        InputStreamReader reader1 = 
            new InputStreamReader(ClassLoader.getSystemResourceAsStream("replay.xml"), "UTF-8");

        InputStreamReader reader2 = new InputStreamReader(new FileInputStream(tmpFile), "UTF-8");
        
        try {
            int sign1;
            do {
                sign1 = reader1.read();
                assertEquals(sign1, reader2.read());
            }
            while (sign1 > -1);
        }
        finally {
            reader1.close();
            reader2.close();
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.util.console.listener.IOutputListener#outputMsg(java.lang.String)
     */
    @Override
    public void outputMsg(String newMessage) {
        outputStream.println(newMessage);
    }

}
