//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import de.ugoe.cs.autoquest.eventcore.IEventTarget;
import de.ugoe.cs.autoquest.eventcore.gui.Scroll;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.tasktrees.treeifc.DefaultTaskInstanceTraversingVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: 16.07.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class TargetDistanceRule implements UsabilityEvaluationRule {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.usability.UsabilityEvaluationRule#evaluate(TaskTree)
     */
    @Override
    public UsabilityEvaluationResult evaluate(ITaskModel taskModel) {
        UsabilityEvaluationResult results = new UsabilityEvaluationResult(taskModel);

        checkForTargetDistances(results, taskModel);

        return results;
    }

    /**
     * 
     */
    private void checkForTargetDistances(UsabilityEvaluationResult results, ITaskModel taskModel) {
        for (ITask task : taskModel.getTasks()) {
            if (task instanceof ISequence) {

                int cummulativeNoOfHops = 0;
                int cummulativeDistance = 0;

                for (ITaskInstance instance : task.getInstances()) {
                    int[] stats = getTargetDistance(instance);
                    cummulativeNoOfHops += stats[0] - 1;
                    cummulativeDistance += stats[1];
                }

                createHighTargetDisanceIfRequired
                    (cummulativeNoOfHops, cummulativeDistance, task, results, taskModel);
            }
        }
    }

    /**
     *
     */
    private int[] getTargetDistance(ITaskInstance instance) {
        List<IEventTarget> eventTargets = new LinkedList<IEventTarget>();
        getEventTargets(instance, eventTargets);
        int noOfGUIElements = eventTargets.size();
        int distance = 0;
        
        while (eventTargets.size() > 1) {
            distance += getDistance(eventTargets.get(0), eventTargets.get(1));
            eventTargets.remove(0);
        }
        
        return new int[] { noOfGUIElements, distance };
    }

    /**
     *
     */
    private int getDistance(IEventTarget eventTarget1, IEventTarget eventTarget2) {
        if ((eventTarget1 instanceof IGUIElement) && (eventTarget2 instanceof IGUIElement)) {
            return (int)
                (1000 * (((IGUIElement) eventTarget1).getDistanceTo((IGUIElement) eventTarget2)));
        }
        else if (eventTarget1.equals(eventTarget2)) {
            return 0;
        }
        else {
            return 1000;
        }
    }

    /**
     *
     */
    private void getEventTargets(ITaskInstance instance, final List<IEventTarget> eventTargets) {
        instance.accept(new DefaultTaskInstanceTraversingVisitor() {
            @Override
            public void visit(IEventTaskInstance eventTaskInstance) {
                if (!(eventTaskInstance.getEvent().getType() instanceof Scroll)) {
                    eventTargets.add(eventTaskInstance.getEvent().getTarget());
                }
            }
        });
    }

    /**
     *
     */
    private void createHighTargetDisanceIfRequired(int                       cummulativeNoOfHops,
                                                   int                       cummulativeDistance,
                                                   ITask                     task,
                                                   UsabilityEvaluationResult results,
                                                   ITaskModel                taskModel)
    {
        if ((cummulativeDistance > 0) && (cummulativeNoOfHops > 0)) {
            int ratio = cummulativeDistance / cummulativeNoOfHops;

            // for HTML: 800 means not even on the same server
            // for HTML: 600 means not on the same page
            // for HTML: 501 means in average not on the same page
            UsabilityDefectSeverity severity = UsabilityDefectSeverity.getSeverity
                (ratio, 800, 600, 501, 501, task, taskModel);

            if (severity != null) {
                double averageNoOfGUIElements =
                    ((double) cummulativeNoOfHops / task.getInstances().size()) + 1;
                
                Map<String, Object> parameters = new HashMap<String, Object>();
                parameters.put("task", task);
                parameters.put("noOfGUIElements", averageNoOfGUIElements);
                parameters.put("distance", ((double) ratio / 1000));

                results.addDefect
                    (severity, UsabilityDefectDescription.HIGH_TARGET_DISTANCE, parameters);
            }
        }
    }

}
