//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.uml;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.Resource.Factory.Registry;
import org.eclipse.emf.ecore.resource.impl.ResourceFactoryImpl;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.resource.UMLResource;
import org.eclipse.uml2.uml.resources.util.UMLResourcesUtil;
import org.junit.After;
import org.junit.Test;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.http.HTTPLogParser;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SOAPEventType;
import de.ugoe.cs.autoquest.plugin.uml.eventcore.UMLTransitionType;

/**
 * <p>
 * Tests for AutoQUESTs UMLUtils
 * </p>
 * 
 * @author Steffen Herbold
 */
public class UMLUtilsTest {
    
    private final static String OUTPUT_DIR = "target/tmp/test-outputs/";
    
    /**
    *
    */
   @After
   public void tearDown() throws Exception {
       deleteFiles(new File(OUTPUT_DIR));
   }

    @Test(expected=java.lang.RuntimeException.class)
    public void createUMLTransitionSequence_1() throws Exception {
        // parse log file
        HTTPLogParser parser = new HTTPLogParser();
        parser.parseFile(new File(ClassLoader.getSystemResource("createSequence_1_usagedata.log")
            .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();
        Model model =
            loadModelFromInputStream(ClassLoader
                .getSystemResourceAsStream("createSequence_1_model.uml"));

        StateMachine stateMachine =
            (StateMachine) model.getPackagedElement("PatientIDBehavior", true,
                                                    UMLPackage.Literals.STATE_MACHINE, true);

        Collection<List<Event>> umlSequences = new LinkedList<>();
        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
                else {
                    if (!event.getType().toString().contains("ixsmq")) {
                        eventIter.remove();
                    }
                }
            }

            umlSequences.add(UMLUtils.createUMLTransitionSequence(httpSequence, stateMachine));
        }
    }
    
    @Test
    public void convertStateMachineToUsageProfile_1() throws Exception {
        // parse log file
        HTTPLogParser parser = new HTTPLogParser();
        parser.parseFile(new File(ClassLoader.getSystemResource("createSequence_1_usagedata.log")
            .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();
        Model model =
            loadModelFromInputStream(ClassLoader
                .getSystemResourceAsStream("createSequence_1_model.uml"));

        StateMachine stateMachine =
            (StateMachine) model.getPackagedElement("PatientIDBehavior", true,
                                                    UMLPackage.Literals.STATE_MACHINE, true);

        Collection<List<Event>> umlSequences = new LinkedList<>();
        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
                else {
                    if (!event.getType().toString().contains("ixsmq")) {
                        eventIter.remove();
                    }
                }
            }

            List<List<Transition>> matchingSequences = UMLUtils.determineMatchingTransitionSequences(httpSequence, stateMachine);
            if (matchingSequences.size() >= 1) {
                List<Event> umlEventSequence = new LinkedList<>();
                for (Transition transition : matchingSequences.get(0)) {
                    umlEventSequence.add(new Event(new UMLTransitionType(transition)));
                }
                umlSequences.add(umlEventSequence);
            }
        }
        UMLUtils.convertStateMachineToUsageProfile(umlSequences, stateMachine);
        
        writeModelToFile(model, OUTPUT_DIR + "convertStateMachineToUsageProfile_1.uml");
    }
    
    private static void writeModelToFile(Model model, String filename) throws IOException {
        final ResourceSet resourceSet = new ResourceSetImpl();
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
            .put(Resource.Factory.Registry.DEFAULT_EXTENSION, new ResourceFactoryImpl() {
                @Override
                public Resource createResource(URI uri) {
                    return new XMIResourceImpl(uri);
                }
            });

        Resource resource = resourceSet.createResource(URI.createURI("binresource"));

        resource.getContents().add(model);
        FileOutputStream fos;
        File file = new File(filename);
        file.getParentFile().mkdirs();
        fos = new FileOutputStream(file);
        resource.save(fos, null);
    }

    private static Model loadModelFromInputStream(InputStream inputStream) {
        ResourceSet resourceSet = new ResourceSetImpl();
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
            .put(UMLResource.FILE_EXTENSION, UMLResource.Factory.INSTANCE);
        resourceSet.getPackageRegistry().put(UMLPackage.eNS_URI, UMLPackage.eINSTANCE);
        resourceSet.getPackageRegistry().put(EcorePackage.eNS_URI, EcorePackage.eINSTANCE);
        resourceSet.getResourceFactoryRegistry();
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
            .put(Registry.DEFAULT_EXTENSION, new XMIResourceFactoryImpl());
        Map<URI, URI> uriMap = resourceSet.getURIConverter().getURIMap();

        UMLResourcesUtil.init(resourceSet);

        uriMap.put(URI.createURI("pathmap://Papyrus.profile.uml"),
                   URI.createURI("file:/D:/.../Papyrus.profile.uml/"));

        EPackage.Registry.INSTANCE.put(UMLPackage.eNS_URI, UMLPackage.eINSTANCE);

        Resource resource = new XMIResourceImpl();
        /*
         * XMIResource.XMLMap xmlMap = new XMLMapImpl(); Map options = new HashMap();
         * options.put(XMIResource.OPTION_XML_MAP, xmlMap);
         */

        try {
            // resource.load(inputStream, options);
            resource.load(inputStream, null);
        }
        catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        return (Model) EcoreUtil.getObjectByType(resource.getContents(), UMLPackage.Literals.MODEL);
    }

   private void deleteFiles(File file) {
       if (file.exists()) {
           if (file.isDirectory()) {
               for (File child : file.listFiles()) {
                   deleteFiles(child);
               }
           }
           
           try {
               file.delete();
           }
           catch (Exception e) {
               // ignore and delete as much as possible
           }
       }
   }

}
