//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.temporalrelation;

import java.util.Iterator;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IOptional;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IOptionalInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequenceInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskBuilder;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstanceList;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskFactory;

/**
 * <p>
 * provides some convenience methods for rule application
 * </p>
 * 
 * @author Patrick Harms
 */
class RuleUtils {

	/**
	 * <p>
	 * counter for generating unique ids. Starts at 0 for each new program start
	 * </p>
	 */
	private static int idCounter = 0;

	/**
	 * <p>
	 * generates a sub sequence for a specified range of elements in the
	 * provided task instances list.
	 * </p>
	 *
	 * @param parent
	 *            the list of which the range shall be extracted
	 * @param startIndex
	 *            the start index of the range
	 * @param endIndex
	 *            the end index of the range (inclusive)
	 * @param model
	 *            the task model (required for instantiating the sub sequence)
	 * @param taskFactory
	 *            the task factory used for instantiating the sub sequence
	 * @param taskBuilder
	 *            the task builder to perform changes in the task structure
	 * 
	 * @return a task instance representing the requested sub sequence
	 */
	static ITaskInstance getSubSequenceInRange(ITaskInstanceList parent,
			int startIndex, int endIndex, ISequence model,
			ITaskFactory taskFactory, ITaskBuilder taskBuilder) {
		ISequenceInstance subsequence = taskFactory
				.createNewTaskInstance(model);

		for (int i = startIndex; i <= endIndex; i++) {
			taskBuilder.addChild(subsequence, parent.get(i));
		}

		return subsequence;
	}

	/**
	 * <p>
	 * replaces a sub sequence for a specified range of elements in the provided
	 * task instances list by a sub task instance
	 * </p>
	 *
	 * @param parent
	 *            the list of which the range shall be replaced
	 * @param startIndex
	 *            the start index of the range
	 * @param endIndex
	 *            the end index of the range (inclusive)
	 * @param model
	 *            the task model (required for instantiating the sub sequence)
	 * @param taskFactory
	 *            the task factory used for instantiating the sub sequence
	 * @param taskBuilder
	 *            the task builder to perform changes in the task structure
	 * 
	 * @return the replacement for the range
	 */
	static ISequenceInstance createNewSubSequenceInRange(
			ITaskInstanceList parent, int startIndex, int endIndex,
			ISequence model, ITaskFactory taskFactory, ITaskBuilder taskBuilder) {
		ISequenceInstance subsequence = taskFactory
				.createNewTaskInstance(model);

		int modelindex=0;
		for (int i = startIndex; i <= endIndex; i++) {
			System.out.println("Trying to add " + parent.get(startIndex)
					+ " to the model instance");
			ITask tempTask = model.getChildren().get(modelindex);
			if (tempTask.getType() == "optionality") {
					IOptionalInstance optional = taskFactory.createNewTaskInstance((IOptional) tempTask);
			} else if (tempTask.getType() == "selection") {

			} else if (tempTask.getType() == "sequence") {
				taskBuilder.addChild(subsequence, parent.get(startIndex));
			} else if (tempTask.getType() == "iteration") {

			} else {

			}
			taskBuilder.removeTaskInstance(parent, startIndex);
			modelindex++;
		}

		taskBuilder.addTaskInstance(parent, startIndex, subsequence);

		return subsequence;
	}

	/**
	 * <p>
	 * returns the next available id (uses the id counter)
	 * </p>
	 * 
	 * @return the next available id
	 */
	static synchronized String getNewId() {
		return Integer.toString(idCounter++);
	}

	/**
	 * <p>
	 * prevent instantiation
	 * </p>
	 */
	private RuleUtils() {
		// prevent instantiation
	}

}
