//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.commands;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.SequenceInstanceOf;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCGUIElement;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to create a Jacareto xml replay file from stored sessions.
 * </p>
 * 
 * @author Daniel May
 * @version 1.0
 */
public class CMDgenerateJacaretoReplay implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "generateJacaretoReplay <filename> <sequences>";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void run(List<Object> parameters) {
        String filename;
        String sequencesName;
        try {
            filename = (String) parameters.get(0);
            sequencesName = (String) parameters.get(1);
        }
        catch (Exception e) {
            throw new IllegalArgumentException();
        }

        Collection<List<Event>> sequences = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(sequencesName);
            return;
        }
        if (!SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            CommandHelpers.objectNotType(sequencesName, "Collection<List<Event<?>>>");
            return;
        }

        sequences = (Collection<List<Event>>) dataObject;

        writeJacaretoXML(sequences, filename);
    }

    private void writeLine(BufferedWriter writer, String line) throws IOException {
        writer.write(line);
        writer.newLine();
    }

    private void writeJacaretoHead(BufferedWriter writer) throws IOException {
        writeLine(writer, "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>");
        writeLine(writer, "<JacaretoStructure>");
        writeLine(writer, "<Record>");

        // TODO: This header content is basically copy+paste from a
        // specific jacareto replay file right now.
        // Some things such as screen resolution and especially the
        // application starter details need to be changed for general cases.
        writeLine(writer,
                  "<Calendar procTime=\"0\" duration=\"0\" year=\"2014\" month=\"8\" date=\"11\" hour=\"14\" min=\"43\" sec=\"41\" uuid=\"06831ba1-f28a-4e05-b46e-ce9d8f9ffa0f\" />");
        writeLine(writer,
                  "<SystemInfo procTime=\"0\" duration=\"0\" screenWidth=\"2646\" screenHeight=\"1024\" javaVersion=\"1.7.0_65\" lookAndFeel=\"javax.swing.plaf.metal.MetalLookAndFeel\" uuid=\"720f430f-52cf-4d8b-9fbe-58434f766efe\" />");
        writeLine(writer,
                  "<KeyboardState procTime=\"0\" duration=\"0\" isNumLockOn=\"false\" isScrollLockOn=\"false\" isCapsLockOn=\"false\" applyIsNumLockOn=\"true\" applyIsScrollLockOn=\"true\" applyIsCapsLockOn=\"true\" uuid=\"28146f79-9fc7-49f9-b4a8-5866a7625683\" />");
        writeLine(writer, "ComponentMode numberPopupMenues=\"true\" />");
        writeLine(writer,
                  "<ApplicationStarter procTime=\"5\" duration=\"5\" name=\"HelloWorldSwing\" class=\"HelloWorldSwing\" initclass=\"\" basepath=\"/home/daniel/project/autoquest-jfcmonitor\" classpathext=\"${basepath}/helloswing.jar;${basepath}/.;\" detectDuration=\"false\" captureparams=\"\" replayparams=\"\" uuid=\"a7b7d7b9-caa9-4d6d-b052-cf74d353275e\" />");
    }

    private ArrayList<String> writeJacaretoEvents(BufferedWriter writer,
                                                  Collection<List<Event>> sequences)
        throws IOException
    {
        ArrayList<String> structure = new ArrayList<String>();
        structure.add("<StructureElement class=\"jacareto.struct.RootElement\">");
        // reference the elements that we included in the header
        structure.add("<Recordable ref=\"1\" />"); // Calendar
        structure.add("<Recordable ref=\"2\" />"); // SystemInfo
        structure.add("<Recordable ref=\"3\" />"); // KeyboardState
        structure.add("<Recordable ref=\"4\" />"); // ApplicationStarter

        for (List<Event> sequence : sequences) {
            for (Iterator<Event> eventIter = sequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();

                // TODO: do a mapping file or something to map
                // from autoquest events to jacareto events
                if (event.getType().getName().equals("KeyboardFocusChange")) {
                    /*
                    writeLine(
                        "<FocusEvent procTime=\"0\" duration=\"0\" source="JFrame_(1).JRootPane_(1).JLayeredPane_(1).JPanel_(1).JPanel_(1).JCheckBox_(1)" +" +
                        "       class="javax.swing.JCheckBox" uuid="061bee8f-d8a3-477f-ab8d-bab02f614916" ID="1004" component="null" root="JFrame_(1)" xPos="0" yPos="0" width="0" height="0" isTemporary="false" />");
                    */
                    System.out.println(((JFCGUIElement)event.getTarget()).getJacaretoHierarchy());
                }
                else {
                    System.out.println(event.getType());
                    System.out.println(event.getTarget());
                }
            }
        }

        return structure;
    }

    private void writeJacaretoTail(BufferedWriter writer, ArrayList<String> structure)
        throws IOException
    {
        writeLine(writer, "</Record>");

        // write the recording's structure
        writeLine(writer, "<Structure>");
        for (String element : structure) {
            writeLine(writer, element);
        }
        writeLine(writer, "</Structure>");
    }

    private void writeJacaretoXML(Collection<List<Event>> sequences, String filename) {
        BufferedWriter writer = new BufferedWriter(openReplayFile(filename + ".xml"));

        try {
            writeJacaretoHead(writer);
            ArrayList<String> structure = writeJacaretoEvents(writer, sequences);
            writeJacaretoTail(writer, structure);
            writeLine(writer, "<JacaretoStructure>");

            writer.flush();
            writer.close();
        }
        catch (IOException e) {
            Console.printerrln("Unable to write Jacareto replay file " + filename);
        }
    }

    /**
     * <p>
     * Helper function that opens the replay file for writing.
     * </p>
     * 
     * @param filename
     *            name and path of the replay file
     * @param encoding
     *            file encoding, empty string for platform default
     * @return {@link OutputStreamWriter} that writes to the replay file
     */
    private OutputStreamWriter openReplayFile(String filename) {
        File file = new File(filename);
        boolean fileCreated;
        try {
            fileCreated = file.createNewFile();
            if (!fileCreated) {
                Console.traceln(Level.INFO, "Created logfile " + filename);
            }
            else {
                Console.traceln(Level.INFO, "Overwrote existing logfile " + filename);
            }
        }
        catch (IOException e) {
            Console.printerrln("Unable to create file " + filename);
            Console.logException(e);
        }
        OutputStreamWriter writer = null;
        try {
            writer = new OutputStreamWriter(new FileOutputStream(file));
        }
        catch (IOException e) {
            Console.printerrln("Unable to open file for writing (read-only file):" + filename);
            Console.logException(e);
        }
        return writer;
    }
}
