//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.logging.Level;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.rules.UsabilityResult;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRule;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRuleset;
import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * The usability evaluator is responsible for performing a usability evaluation on a task model.
 * Therefore he uses a {@link UsabilityRuleset}.
 * </p>
 * 
 * @author Alexander Deicke
 */
public class UsabilityEvaluator {

    /**
     * <p>
     * The task model, which should be evaluated.
     * </p>
     */
    private ITaskModel taskModel;

    private UsabilityEvaluator(ITaskModel taskModel) {
        this.taskModel = taskModel;
    }

    /**
     * <p>
     * Creates and initializes a new instance.
     * </p>
     * 
     * @param taskModel
     *            task model, which is subject of usability evaluation
     * @return instance of {@link UsabilityEvaluator}
     */
    public static UsabilityEvaluator evaluate(ITaskModel taskModel) {
        return new UsabilityEvaluator(taskModel);
    }

    /**
     * <p>
     * Starts usability evaluation, using given {@link UsabilityRuleset}.
     * </p>
     * 
     * @param ruleset
     *            {@link UsabilityRuleset} used for usability evaluation
     * @return result of usability evaluation as {@link UsabilityResult}
     */
    public UsabilityResult using(UsabilityRuleset ruleset) {
        Console.traceln(Level.INFO, "evaluating usability of task tree " + this.taskModel);
        EvaluationMethodCaller evaluationMethodCaller = new EvaluationMethodCaller();
        UsabilityResult result = new UsabilityResult();
        for (UsabilityRule rule : ruleset.evaluationRules()) {
            Optional<UsabilityProblemDescription> defect =
                rule.callEvaluationMethod(evaluationMethodCaller);
            if (defect.isPresent()) {
                UsabilityProblemDescription description = defect.get();
                logUsabilityProblem(description);
                result.addProblem(description);
            }
        }
        return result;
    }

    private void logUsabilityProblem(UsabilityProblemDescription description) {
        Level level = null;
        switch (description.getSeverityLevel()) {
            case NONE:
                break;
            case INFO:
            case LOW:
                level = Level.INFO;
                break;
            case MEDIUM:
            case HIGH:
                level = Level.WARNING;
        }
        if (level != null) {
            Console.print(description.getSeverityLevel().toString());
            Console.print(" : ");
            Console.println(description.toString());
            //Console.traceln(level, description.toString());
        }
    }

}
