//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.htmlmonitor;

/**
 * <p>
 * represents an event caused by a user on a specific web site. An event contains the infos
 * about the client ({@link HtmlClientInfos}, when ant where the event took place, the type of
 * event and some additional infos such as the event coordinates or the number of the pressed
 * key. 
 * </p>
 * 
 * @author Patrick Harms
 */
class HtmlEvent {

    /**
     * infos about the client that caused the event
     */
    private HtmlClientInfos clientInfos;

    /**
     * the time stamp of the event
     */
    private Long time;

    /**
     * the HTML element on which the event was executed
     */
    private HtmlPageElement target;

    /**
     * the document to which the HTML element on which the event was executed belongs
     */
    private HtmlDocument targetDocument;

    /**
     * the targets DOM path through the document to which it belongs
     */
    private String targetDOMPath;

    /**
     * the type of the event, e.g. onclick
     */
    private String eventType;

    /**
     * the coordinates of the event, usually an array with two values (x and y)
     */
    private Integer[] coordinates;

    /**
     * if the event is a key event, the key that was pressed or released
     */
    private Integer key;

    /**
     * if the event is a scroll event, the resulting position of the scrolled element
     */
    private Integer[] scrollPosition;

    /**
     * if the event is an on change event, the value to which the changed element is changed
     */
    private String selectedValue;

    /**
     * <p>
     * initializes the event with all relevant infos
     * </p>
     *
     * @param clientInfos    infos about the client that caused the event
     * @param time           the time stamp of the event
     * @param target         the HTML element on which the event was executed
     * @param eventType      the type of the event, e.g. onclick
     * @param coordinates    the coordinates of the event, usually an array with two values
     *                       (x and y)
     * @param key            if the event is a key event, the key that was pressed or released
     * @param scrollPosition if the event is a scroll event, the resulting position of the
     *                       scrolled element
     * @param selectedValue  if the event is an on change event, the value to which the changed
     *                       element is changed
     */
    HtmlEvent(HtmlClientInfos clientInfos,
              Long            time,
              HtmlPageElement target,
              String          eventType,
              Integer[]       coordinates,
              Integer         key,
              Integer[]       scrollPosition,
              String          selectedValue)
    {
        this.clientInfos = clientInfos;
        this.time = time;
        this.target = target;
        this.targetDocument = target.getDocument();
        this.targetDOMPath = target.getDOMPath();
        this.eventType = eventType;
        this.coordinates = coordinates;
        this.key = key;
        this.scrollPosition = scrollPosition;
        this.selectedValue = selectedValue;
    }

    /**
     * <p>
     * initializes the event for which the id of the target is not known yet. In this case
     * the document and DOM path for the target are provided
     * </p>
     *
     * @param clientInfos    infos about the client that caused the event
     * @param time           the time stamp of the event
     * @param targetDocument the document to which the HTML element belongs on which the event was
     *                       executed
     * @param targetDOMPath  the path through the DOM of the document of the HTML element on which
     *                       the event was executed
     * @param eventType      the type of the event, e.g. onclick
     * @param coordinates    the coordinates of the event, usually an array with two values
     *                       (x and y)
     * @param key            if the event is a key event, the key that was pressed or released
     * @param scrollPosition if the event is a scroll event, the resulting position of the
     *                       scrolled element
     * @param selectedValue  if the event is an on change event, the value to which the changed
     *                       element is changed
     */
    HtmlEvent(HtmlClientInfos clientInfos,
              Long            time,
              HtmlDocument    targetDocument,
              String          targetDOMPath,
              String          eventType,
              Integer[]       coordinates,
              Integer         key,
              Integer[]       scrollPosition,
              String          selectedValue)
    {
        this.clientInfos = clientInfos;
        this.time = time;
        this.targetDocument = targetDocument;
        this.targetDOMPath = targetDOMPath;
        this.eventType = eventType;
        this.coordinates = coordinates;
        this.key = key;
        this.scrollPosition = scrollPosition;
        this.selectedValue = selectedValue;
    }

    /**
     * @return the clientInfos
     */
    HtmlClientInfos getClientInfos() {
        return clientInfos;
    }

    /**
     * @return the time
     */
    Long getTime() {
        return time;
    }

    /**
     * @return the target
     */
    HtmlPageElement getTarget() {
        return target;
    }

    /**
     * @return the targetDocument
     */
    HtmlDocument getTargetDocument() {
        return targetDocument;
    }

    /**
     * @return the targetDOMPath
     */
    String getTargetDOMPath() {
        return targetDOMPath;
    }

    /**
     * @return the eventType
     */
    String getEventType() {
        return eventType;
    }

    /**
     * @return the coordinates
     */
    Integer[] getCoordinates() {
        return coordinates;
    }

    /**
     * @return the key
     */
    Integer getKey() {
        return key;
    }

    /**
     * @return the scrollPosition
     */
    Integer[] getScrollPosition() {
        return scrollPosition;
    }

    /**
     * @return the selectedValue
     */
    String getSelectedValue() {
        return selectedValue;
    }

}
