//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.commands;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.plugin.jfc.JFCSimplifiedLogParser;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCGUIElementSpec;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to generate sessions in which the JFCGUIElements contain a special index that is required
 * to replay the sequences with Jacareto.
 * </p>
 * 
 * @author Daniel May
 * @version 1.0
 */
public class CMDparseJFCwithJacaretoIndices implements Command {

    @Override
    public void run(List<Object> parameters) {
        String filename;
        String sequencesName = "sequences";
        try {
            filename = (String) parameters.get(0);
            if (parameters.size() >= 2) {
                sequencesName = (String) parameters.get(1);
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException();
        }

        JFCSimplifiedLogParser parser = new JFCSimplifiedLogParser();

        try {
            parser.parseFile(filename);
        }
        catch (Exception e) {
            Console.printerrln("Could not parse " + filename + ": " + e.getMessage());
            return;
        }

        Collection<List<Event>> sequences = parser.getSequences();
        GUIModel targets = parser.getGuiModel();

        generateJacaretoIndices(targets.getRootElements(), targets);

        if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
            CommandHelpers.dataOverwritten(sequencesName);
        }

        if (GlobalDataContainer.getInstance().addData(sequencesName + "_targets", targets)) {
            CommandHelpers.dataOverwritten(sequencesName + "_targets");
        }
    }

    @Override
    public String help() {
        return "parseJFCwithJacaretoIndices <filename> {<sequences>}";
    }

    private void generateJacaretoIndices(List<IGUIElement> elements, GUIModel targets) {
        HashMap<String, Integer> typeCount = new HashMap<>();

        for (IGUIElement child : elements) {
            String type = child.getSpecification().getType();
            Integer count = typeCount.get(type);

            if (count == null) {
                count = 0;
                typeCount.put(type, count);
            }
            else {
                typeCount.put(type, ++count);
            }

            ((JFCGUIElementSpec) child.getSpecification()).setAltIndex(count);

            generateJacaretoIndices(targets.getChildren(child), targets);
        }
    }
}
