//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.usability;

import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.UsabilityEvaluationManager;
import de.ugoe.cs.autoquest.usability.UsabilityEvaluationResult;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * This command performs a usability evaluation based on a task tree. It uses the
 * {@link UsabilityEvaluationManager} for this purpose. Please consult the documentation of the
 * usability evaluation manager for more details.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDevaluateUsability implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "evaluateUsability <tasktree> {<evaluationResult>}";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String tasktreeName;
        String evaluationResult;
        try {
            tasktreeName = (String) parameters.get(0);
            if (parameters.size() > 1) {
                evaluationResult = (String) parameters.get(1);
            }
            else {
                evaluationResult = "usabilityEvaluationResult";
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("must provide a task tree name");
        }

        Object dataObject = GlobalDataContainer.getInstance().getData(tasktreeName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(tasktreeName);
            return;
        }
        if (!(dataObject instanceof ITaskModel)) {
            CommandHelpers.objectNotType(tasktreeName, "ITaskTree");
            return;
        }

        ITaskModel taskTree = (ITaskModel) dataObject;
        
        UsabilityEvaluationResult result =
            new UsabilityEvaluationManager().evaluateUsability(taskTree);
        
        if (GlobalDataContainer.getInstance().addData(evaluationResult, result)) {
            CommandHelpers.dataOverwritten(evaluationResult);
        }
        
    }

}
