//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelectionInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;

/**
 * <p>
 * this task comparison rule is capable of comparing selections. If both
 * selections do not have children, they are treated as lexically equal. If they
 * have children, each child of both selections is compared to each child of the
 * respective other selection. The resulting equality is the most concrete one
 * of all these comparisons. I.e. if all children are at least lexically equal,
 * then the selections are lexically equal. If all children are at least
 * syntactically equal, then the selections are syntactically equal. If all
 * children are at least semantically equal, then the selections are
 * semantically equal. If only one of the selections has children, then the
 * selections are unequal. The comparison is broken up, if only a specific
 * equality is checked for and this equality is ensured.
 * </p>
 * 
 * @version $Revision: $ $Date: 19.02.2012$
 * @author 2012, last modified by $Author: patrick$
 */
public class SelectionComparisonRule implements TaskComparisonRule {

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areLexicallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areLexicallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.LEXICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areLexicallyEqual(ITaskInstance, ITaskInstance)
	 */
	@Override
	public boolean areLexicallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.LEXICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSemanticallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areSemanticallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.SEMANTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSemanticallyEqual(ITaskInstance,
	 * ITaskInstance)
	 */
	@Override
	public boolean areSemanticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.SEMANTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSyntacticallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.SYNTACTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSyntacticallyEqual(ITaskInstance,
	 * ITaskInstance)
	 */
	@Override
	public boolean areSyntacticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.SYNTACTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
	}

	/**
	 * <p>
	 * used to to call the task equality rule manager for the comparison of the
	 * two provided children. If no required equality level is provided, than
	 * the most concrete equality is returned. Otherwise, the required equality
	 * is returned as long as the children are equal on that level.
	 * </p>
	 * 
	 * @param child1
	 *            the first task to be compared
	 * @param child2
	 *            the second task to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality
	 */
	private TaskEquality callRuleManager(ITask child1, ITask child2,
			TaskEquality requiredEqualityLevel) {
		if (requiredEqualityLevel == null) {
			return TaskEqualityRuleManager.getInstance()
					.compare(child1, child2);
		} else if (TaskEqualityRuleManager.getInstance().areAtLeastEqual(
				child1, child2, requiredEqualityLevel)) {
			return requiredEqualityLevel;
		} else {
			return TaskEquality.UNEQUAL;
		}
	}

	/**
	 * <p>
	 * used to to call the task equality rule manager for the comparison of the
	 * two provided children. If no required equality level is provided, than
	 * the most concrete equality is returned. Otherwise, the required equality
	 * is returned as long as the children are equal on that level.
	 * </p>
	 * 
	 * @param taskInstance1
	 *            the first task instance to be compared
	 * @param taskInstance2
	 *            the second task instance to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality
	 */
	private TaskEquality callRuleManager(ITaskInstance taskInstance1,
			ITaskInstance taskInstance2, TaskEquality requiredEqualityLevel) {
		if (requiredEqualityLevel == null) {
			return TaskEqualityRuleManager.getInstance().compare(taskInstance1,
					taskInstance2);
		} else if (TaskEqualityRuleManager.getInstance().areAtLeastEqual(
				taskInstance1, taskInstance2, requiredEqualityLevel)) {
			return requiredEqualityLevel;
		} else {
			return TaskEquality.UNEQUAL;
		}
	}

	/**
	 * <p>
	 * ensures for the two given lists, that for at least one task in the first
	 * list there is a task in the second list being on the given level equal to
	 * the task in the first list.
	 * </p>
	 * 
	 * @param children1
	 *            the first list to be compared
	 * @param children2
	 *            the second list to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return true if there is a task in the first list that has an equal task
	 *         in the second list when considering the given equality level,
	 *         false else.
	 */
	private boolean checkEqualityLevel(List<ITask> children1,
			List<ITask> children2, TaskEquality requiredEqualityLevel) {
		TaskEquality currentEquality;
		for (final ITask child1 : children1) {
			for (final ITask child2 : children2) {
				currentEquality = callRuleManager(child1, child2,
						requiredEqualityLevel);
				if ((currentEquality != null)
						&& (currentEquality.isAtLeast(requiredEqualityLevel))) {
					// we found at least one equal child with sufficient
					// equality in the
					// second list. So be can break up for this child.
					return true;
				}
			}
		}

		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#compare(ITask, ITask)
	 */
	@Override
	public TaskEquality compare(ITask task1, ITask task2) {
		return getEquality(task1, task2, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#compare(ITaskInstance, ITaskInstance)
	 */
	@Override
	public TaskEquality compare(ITaskInstance instance1, ITaskInstance instance2) {
		return getEquality(instance1, instance2, null);
	}

	/**
	 * <p>
	 * compares two selections with each other checking for the provided
	 * required level of equality. If this level is ensured, the method
	 * immediately returns. The more concrete the required equality level, the
	 * more checks this method performs.
	 * </p>
	 * 
	 * @param task1
	 *            the first task to be compared
	 * @param task2
	 *            the second task to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality.
	 */
	private TaskEquality getEquality(ITask task1, ITask task2,
			TaskEquality requiredEqualityLevel) {
		final List<ITask> children1 = ((ISelection) task1).getChildren();
		final List<ITask> children2 = ((ISelection) task2).getChildren();

		// if both selections do not have children, they are lexically equal. If
		// only one of them
		// has children, they are unequal.
		if ((children1.size() == 0) && (children2.size() == 0)) {
			return TaskEquality.LEXICALLY_EQUAL;
		} else if ((children1.size() == 0) || (children2.size() == 0)) {
			return TaskEquality.UNEQUAL;
		}

		if (requiredEqualityLevel == null) {
			// calculate the common equality level for all children of both
			// selections.
			// do it in both directions to ensure commutative comparison
			return getMostConcreteEqualityLevel(children1, children2);
		} else {
			// we are searching for a specific equality
			if (checkEqualityLevel(children1, children2, requiredEqualityLevel)) {
				return requiredEqualityLevel;
			} else {
				return TaskEquality.UNEQUAL;
			}
		}
	}

	/**
	 * <p>
	 * compares two selection instances with each other checking for the
	 * provided required level of equality. If this level is ensured, the method
	 * immediately returns. The more concrete the required equality level, the
	 * more checks this method performs.
	 * </p>
	 * 
	 * @param taskInstance1
	 *            the first task instance to be compared
	 * @param taskInstance2
	 *            the second task instance to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality.
	 */
	private TaskEquality getEquality(ITaskInstance taskInstance1,
			ITaskInstance taskInstance2, TaskEquality requiredEqualityLevel) {
		final ITaskInstance child1 = ((ISelectionInstance) taskInstance1)
				.getChild();
		final ITaskInstance child2 = ((ISelectionInstance) taskInstance2)
				.getChild();

		// if both selections do not have children, they are lexically equal. If
		// only one of them
		// has children, they are unequal.
		if ((child1 == null) && (child2 == null)) {
			return TaskEquality.LEXICALLY_EQUAL;
		} else if ((child1 == null) || (child2 == null)) {
			return TaskEquality.UNEQUAL;
		}

		final TaskEquality equality = callRuleManager(child1, child2,
				requiredEqualityLevel);

		if (equality == TaskEquality.IDENTICAL) {
			// two different selection instances can be at most lexically equal
			// even if their
			// children are identical
			return TaskEquality.LEXICALLY_EQUAL;
		} else {
			return equality;
		}
	}

	/**
	 * <p>
	 * determines the most concrete equality level for all tasks in the first
	 * list compared to all tasks in the second list. It is sufficient, if there
	 * is one task in one list for which there exist an equal task in the other
	 * list.
	 * </p>
	 *
	 * @param children1
	 *            the first list to be compared
	 * @param children2
	 *            the second list to be compared
	 * 
	 * @return the most concrete task equality identified for all tasks in the
	 *         first list with respect to the second list
	 */
	private TaskEquality getMostConcreteEqualityLevel(List<ITask> children1,
			List<ITask> children2) {
		TaskEquality childEquality;
		TaskEquality currentEquality;
		for (final ITask child1 : children1) {
			childEquality = null;
			for (final ITask child2 : children2) {
				currentEquality = callRuleManager(child1, child2, null);
				if ((currentEquality != null)
						&& (currentEquality != TaskEquality.UNEQUAL)) {
					if (childEquality == null) {
						childEquality = currentEquality;
					} else {
						childEquality = childEquality
								.getCommonDenominator(currentEquality);
					}

					if (childEquality.isAtLeast(TaskEquality.LEXICALLY_EQUAL)) {
						// as we calculate the most concrete equality, we can
						// break up here
						return TaskEquality.LEXICALLY_EQUAL;
					}
				}
			}
		}

		// as the comparison should be commutative, we do not need to check, if
		// in list 2 there is
		// a child equal to one in list 1
		return TaskEquality.UNEQUAL;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#isApplicable(ITask, ITask)
	 */
	@Override
	public boolean isApplicable(ITask task1, ITask task2) {
		return (task1 instanceof ISelection) && (task2 instanceof ISelection);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#isApplicable(ITaskInstance, ITaskInstance)
	 */
	@Override
	public boolean isApplicable(ITaskInstance instance1, ITaskInstance instance2) {
		return isApplicable(instance1.getTask(), instance2.getTask());
	}
}
