//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeimpl;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstanceVisitor;

/**
 * <p>
 * this is the default implementation of the interface {@link ITaskInstance}. It
 * does not do anything fancy except implementing the interface.
 * </p>
 * 
 * @author Patrick Harms
 */
class TaskInstance implements ITaskInstance {

	/**
	 * <p>
	 * creates a new id for a task instance using {@link #temporalId} by
	 * incrementing it an returning its current value. Resets the counter if
	 * {@link Integer.MAX_VALUE} is reached.
	 * </p>
	 * 
	 * @return a new unique id for a task instance as long as
	 *         {@link #temporalId} does not overflow
	 */
	private static synchronized int getNewId() {
		if (temporalId == Integer.MAX_VALUE) {
			temporalId = 0;
		}

		return temporalId++;
	}

	/**
	 * <p>
	 * default serial version UID
	 * </p>
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * <p>
	 * used as a counter to generate new ids for each newly created task
	 * instance. May overflow.
	 * </p>
	 */
	private static int temporalId = 0;

	/**
	 * <p>
	 * the task instantiated by this task instance
	 * </p>
	 */
	private ITask task;

	/**
	 * <p>
	 * the id of the task instance (unique throughout the system as long as
	 * {@link #temporalId} does not overflow.
	 * </p>
	 */
	private final int id;

	/**
	 * <p>
	 * instantiated the task instance with the task that is instantiated by the
	 * instance. It also assigns a unique id to the instance using
	 * {@link #getNewId()}.
	 * </p>
	 */
	TaskInstance(ITask task) {
		this.task = task;
		id = getNewId();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance#accept(
	 * ITaskInstanceVisitor)
	 */
	@Override
	public void accept(ITaskInstanceVisitor visitor) {
		visitor.visit(this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#clone()
	 */
	@Override
	public synchronized ITaskInstance clone() {
		TaskInstance clone = null;
		try {
			clone = (TaskInstance) super.clone();
		} catch (final CloneNotSupportedException e) {
			// this should never happen. Therefore simply dump the exception
			e.printStackTrace();
		}

		return clone;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.ugoe.cs.autoquest.tasktrees.treeifc.TaskTreeNode#equals(TaskTreeNode)
	 */
	@Override
	public boolean equals(ITaskInstance taskInstance) {
		// task instances are only equal if they are identical or if they have
		// the same id
		// (may happen, if they are cloned)
		return (this == taskInstance)
				|| (this.hashCode() == taskInstance.hashCode());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance#getTask()
	 */
	@Override
	public ITask getTask() {
		return task;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public synchronized int hashCode() {
		return id;
	}

	/**
	 * <p>
	 * used to update the task represented through this instance
	 * </p>
	 * 
	 * @param task
	 *            the task to set
	 */
	void setTask(ITask task) {
		this.task = task;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public synchronized String toString() {
		final StringBuffer result = new StringBuffer();
		result.append(task.getType());
		result.append(" #");
		result.append(task.getId());

		if (task.getDescription() != null) {
			result.append(" (");
			result.append(task.getDescription());
			result.append(')');
		}

		/*
		 * if (children != null) { result.append(", ");
		 * result.append(children.size()); result.append(" children"); }
		 */

		return result.toString();
	}

}
