//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.temporalrelation;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.usageprofiles.SymbolMap;

/**
 * <p>
 * symbol map implementation for task instances considering two task instances
 * as equal if their tasks are identical
 * </p>
 * 
 * @author Patrick Harms
 */
public class TaskSymbolIdentityMap<V> implements SymbolMap<ITaskInstance, V> {

	/**  */
	private static final long serialVersionUID = 1L;

	/**
	 * <p>
	 * internally used map for implementing the symbol map interface
	 * </p>
	 */
	private final Map<ITask, V> delegate;

	/**
	 * <p>
	 * mapping between the tasks and the real symbols stored in the map, i.e.
	 * the task instances
	 * </p>
	 */
	private final Map<ITask, ITaskInstance> symbols;

	/**
	 * <p>
	 * initializes this map
	 * </p>
	 */
	public TaskSymbolIdentityMap() {
		delegate = new HashMap<ITask, V>();
		symbols = new HashMap<ITask, ITaskInstance>();
	}

	/**
	 * <p>
	 * copy constructor
	 * </p>
	 *
	 * @param other
	 *            the map to be copied
	 */
	public TaskSymbolIdentityMap(SymbolMap<ITaskInstance, V> other) {
		if (other == null) {
			throw new IllegalArgumentException("other map must not be null");
		}

		delegate = new HashMap<ITask, V>();
		symbols = new HashMap<ITask, ITaskInstance>();

		for (final ITaskInstance symbol : other.getSymbols()) {
			delegate.put(symbol.getTask(), other.getValue(symbol));
			symbols.put(symbol.getTask(), symbol);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.ugoe.cs.autoquest.usageprofiles.SymbolMap#addSymbol(java.lang.Object,
	 * java.lang.Object)
	 */
	@Override
	public void addSymbol(ITaskInstance symbol, V value) {
		if (symbol == null) {
			throw new IllegalArgumentException("symbol must not be null");
		}

		delegate.put(symbol.getTask(), value);
		symbols.put(symbol.getTask(), symbol);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#clear()
	 */
	@Override
	public void clear() {
		delegate.clear();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.ugoe.cs.autoquest.usageprofiles.SymbolMap#containsSymbol(java.lang
	 * .Object)
	 */
	@Override
	public boolean containsSymbol(ITaskInstance symbol) {
		if (symbol == null) {
			throw new IllegalArgumentException("symbol must not be null");
		}

		return delegate.containsKey(symbol.getTask());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		} else if (this.getClass().isInstance(obj)) {
			return delegate.equals(((TaskSymbolIdentityMap<V>) obj).delegate);
		} else {
			return false;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#getSymbols()
	 */
	@Override
	public Collection<ITaskInstance> getSymbols() {
		return symbols.values();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.ugoe.cs.autoquest.usageprofiles.SymbolMap#getValue(java.lang.Object)
	 */
	@Override
	public V getValue(ITaskInstance symbol) {
		if (symbol == null) {
			throw new IllegalArgumentException("symbol must not be null");
		}

		return delegate.get(symbol.getTask());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#getValues()
	 */
	@Override
	public Collection<V> getValues() {
		return delegate.values();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return delegate.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#isEmpty()
	 */
	@Override
	public boolean isEmpty() {
		return delegate.isEmpty();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.ugoe.cs.autoquest.usageprofiles.SymbolMap#removeSymbol(java.lang.Object
	 * )
	 */
	@Override
	public V removeSymbol(ITaskInstance symbol) {
		if (symbol == null) {
			throw new IllegalArgumentException("symbol must not be null");
		}

		symbols.remove(symbol.getTask());
		return delegate.remove(symbol.getTask());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.usageprofiles.SymbolMap#size()
	 */
	@Override
	public int size() {
		return delegate.size();
	}

}
