//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeimpl;

import java.util.LinkedList;
import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IStructuringTemporalRelationship;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;

/**
 * <p>
 * this is the default implementation of the interface
 * {@link IStructuringTemporalRelationship}. It does not do anything fancy
 * except implementing the interface.
 * </p>
 * 
 * @author Patrick Harms
 */
abstract class StructuringTemporalRelationship extends Task implements
		IStructuringTemporalRelationship {

	/**
	 * <p>
	 * default serial version UID
	 * </p>
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * <p>
	 * the list of children of this temporal relationship
	 * </p>
	 */
	private List<ITask> children = new LinkedList<ITask>();

	/**
	 * <p>
	 * initializes this temporal relationship with a human readable name
	 * </p>
	 *
	 * @param relationshipType
	 *            the human readable name of this temporal relationship
	 */
	StructuringTemporalRelationship(String relationshipType) {
		super(relationshipType);

		if ((relationshipType == null) || ("".equals(relationshipType))) {
			throw new IllegalArgumentException(
					"the relationship type must be something meaningful");
		}
	}

	/**
	 * <p>
	 * used to add a new child to this temporal relationship at a specific
	 * position.
	 * </p>
	 *
	 * @param index
	 *            the index of the new child to be added.
	 * @param newChild
	 *            the new child to be added
	 */
	void addChild(int index, ITask newChild) {
		children.add(index, newChild);
	}

	/**
	 * <p>
	 * used to add a new child to this temporal relationship.
	 * </p>
	 *
	 * @param newChild
	 *            the new child to be added
	 */
	void addChild(ITask newChild) {
		children.add(newChild);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.tasktrees.treeimpl.Task#clone()
	 */
	@Override
	public synchronized StructuringTemporalRelationship clone() {
		StructuringTemporalRelationship clone = null;
		clone = (StructuringTemporalRelationship) super.clone();

		clone.children = new LinkedList<ITask>();

		for (final ITask child : this.children) {
			clone.children.add(child.clone());
		}

		return clone;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * de.ugoe.cs.autoquest.tasktrees.treeifc.IStructuringTemporalRelationship
	 * #getChildren()
	 */
	@Override
	public List<ITask> getChildren() {
		return children;
	}

	/**
	 * <p>
	 * removes a child from this temporal relationship at a specific position.
	 * </p>
	 *
	 * @param index
	 *            the index of the child to be removed.
	 */
	void removeChild(int index) {
		children.remove(index);
	}
}
