//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.eventcore;

import java.awt.event.FocusEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;

/**
 * <p>
 * Enumeration to deal with JFC event ids.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public enum JFCEventId {

    MOUSE_CLICKED(MouseEvent.MOUSE_CLICKED), MOUSE_PRESSED(MouseEvent.MOUSE_PRESSED),
    MOUSE_RELEASED(MouseEvent.MOUSE_RELEASED), MOUSE_MOVED(MouseEvent.MOUSE_MOVED), MOUSE_ENTERED(
        MouseEvent.MOUSE_ENTERED), MOUSE_EXITED(MouseEvent.MOUSE_EXITED), MOUSE_DRAGGED(
        MouseEvent.MOUSE_DRAGGED), MOUSE_WHEEL(MouseEvent.MOUSE_WHEEL), FOCUS_GAINED(
        FocusEvent.FOCUS_GAINED), FOCUS_LOST(FocusEvent.FOCUS_LOST), KEY_TYPED(KeyEvent.KEY_TYPED),
    KEY_PRESSED(KeyEvent.KEY_PRESSED), KEY_RELEASED(KeyEvent.KEY_RELEASED);

    /**
     * <p>
     * Numerical representation of the event type.
     * </p>
     */
    private int mNumber;

    /**
     * <p>
     * Constructor. Creates a new JFCEventId.
     * </p>
     * 
     * @param number
     *            numerical representation of the event type.
     */
    JFCEventId(int number) {
        mNumber = number;
    }

    /**
     * <p>
     * Returns the numerical representation of the event type.
     * </p>
     * 
     * @return the numerical representation
     */
    public int getNumber() {
        return mNumber;
    }

    /**
     * <p>
     * Parses an {@link String} and returns the respective JFCEventId if possible.
     * </p>
     * 
     * @param numberString
     *            String representation of the event type
     * @return created JFCEventId
     * @throws IllegalArgumentException
     *             thrown if there is no JFCEventId that correlates to numberString
     */
    public static JFCEventId parseEventId(String numberString) throws IllegalArgumentException {
        try {
            int number = Integer.parseInt(numberString);
            return valueOf(number);
        }
        catch (NumberFormatException e) {
            return JFCEventId.valueOf(JFCEventId.class, numberString);
        }
    }

    /**
     * <p>
     * Returns the JFCEventId associated with an integer.
     * </p>
     * 
     * @param number
     *            integer to which the according JFCEventId is returned
     * @return the JFCEventId
     * @throws IllegalArgumentException
     *             thrown if there is no JFCEventId that correlates to number
     */
    public static JFCEventId valueOf(int number) throws IllegalArgumentException {
        for (JFCEventId type : JFCEventId.values()) {
            if (type.mNumber == number) {
                return type;
            }
        }

        throw new IllegalArgumentException("there is no event type with number " + number);
    }

}
