//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.jfcmonitor;

import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.accessibility.AccessibleContext;

import de.ugoe.cs.util.StringTools;

/**
 * <p>
 * Listener that listens on the name change of a component.
 * </p>
 * 
 * @author Fabian Glaser
 * @version 1.0
 */
public class JFCTitleChangeListener implements PropertyChangeListener {
	/**
	 * <p>
	 * Writer for logging events.
	 * </p>
	 */
	final private JFCMonitorOutputWriter outputWriter;

	final static private boolean CHECK_HIERARCHY = true;

	/**
	 * <p>
	 * Constructor. Creates a new JFCNameChangeListener with a given
	 * {@link JFCMonitorOutputWriter}.
	 * </p>
	 * 
	 * @param outputWriter
	 */
	public JFCTitleChangeListener(JFCMonitorOutputWriter outputWriter) {
		this.outputWriter = outputWriter;
	}

	@Override
	public synchronized void propertyChange(PropertyChangeEvent evt) {
		String propertyName = evt.getPropertyName();
		Component component = null;
		int titleSource = JFCComponentTitleHierachy.SOURCE_NOT_KNOWN;

		// Set source component and title source
		if (propertyName.equals("AccessibleName")) {
			AccessibleContext context = (AccessibleContext) evt.getSource();
			component = (Component) context.getAccessibleParent();
			titleSource = JFCComponentTitleHierachy.ACCESSIBLE_NAME;
		}

		if (propertyName.equals("name")) {
			component = (Component) evt.getSource();
			titleSource = JFCComponentTitleHierachy.NAME;
		}

		if (propertyName.equals("icon")) {
			component = (Component) evt.getSource();
			titleSource = JFCComponentTitleHierachy.ICON;
		}

		if (propertyName.equals("x") || propertyName.equals("y")) {
			component = (Component) evt.getSource();
			titleSource = JFCComponentTitleHierachy.POS;
		}

		if (component != null) {
			JFCComponent jfcComponent = JFCComponent.find(component);
			if (jfcComponent != null) {
				int oldTitleSource = jfcComponent.getTitleSource();
				Object newTitle = evt.getNewValue();
				// We only print the title change message if the new title is as
				// least as informative as the initial title
				if ((!CHECK_HIERARCHY || oldTitleSource <= titleSource)) {
					StringBuilder builder = new StringBuilder();
					builder.append("<componentTitleChange hash=\"");
					builder.append(Integer.toHexString(component.hashCode()));
					builder.append("\" newTitle=\"" + newTitle);
					builder.append("\" titleSource=\"" + titleSource);
					builder.append("\" />" + StringTools.ENDLINE);
					outputWriter.write(builder.toString());
				}
			} else {
				throw new AssertionError("Referenced component is not known.");
			}
		}
	}

}
