//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.usage;

import java.util.List;
import java.util.Random;

import de.ugoe.cs.autoquest.usageprofiles.PredictionByPartialMatch;
import de.ugoe.cs.autoquest.usageprofiles.TrieBasedModel;

/**
 * <p>
 * Command that trains Prediction by Partial Match (PPM) models.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 2.0
 */
public class CMDtrainPPM extends AbstractTrainCommand {

	/**
	 * <p>
	 * Escape probability of the PPM model.
	 * </p>
	 */
	double probEscape;

	/**
	 * <p>
	 * Maximal Markov order of the PPM model.
	 * </p>
	 */
	int maxOrder;

	/**
	 * <p>
	 * Minimal Markov order of the PPM model. Default: 0
	 * </p>
	 */
	int minOrder = 0;

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#help()
	 */
	@Override
	public String help() {
		return "trainPPM <modelname> <sequencesName> <probEscape> <maxOrder> {<minOrder>}";
	}

	/**
	 * <p>
	 * Handles the parameters probEscape, maxOrder, and minOrder.
	 * </p>
	 * 
	 * @see de.ugoe.cs.autoquest.commands.usage.AbstractTrainCommand#handleOptionalParameters(java.util.List)
	 */
	@Override
	void handleAdditionalParameters(List<Object> parameters) throws Exception {
		probEscape = Double.parseDouble((String) parameters.get(2));
		maxOrder = Integer.parseInt((String) parameters.get(3));
		if (parameters.size() == 5) {
			minOrder = Integer.parseInt((String) parameters.get(4));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.autoquest.ui.commands.AbstractTrainCommand#createModel()
	 */
	@Override
	TrieBasedModel createModel() {
		return new PredictionByPartialMatch(maxOrder, minOrder, new Random(),
				probEscape);
	}

}
