//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequenceInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;

// TODO: Auto-generated Javadoc
/**
 * <p>
 * This rule is capable of comparing sequences. If both sequences do not have
 * children, they are treated as lexically equal. Sequences are lexically equal,
 * if they have the same number and order of lexically equal children. The rule
 * can not decide, if two sequences are syntactically or semantically equal.
 * </p>
 *
 * @author 2012, last modified by $Author: patrick$
 * @version $Revision: $ $Date: 19.02.2012$
 */
public class SequenceComparisonRule implements TaskComparisonRule {

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areLexicallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areLexicallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.LEXICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areLexicallyEqual(ITaskInstance, ITaskInstance)
	 */
	@Override
	public boolean areLexicallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.LEXICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSemanticallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areSemanticallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.SEMANTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSemanticallyEqual(ITaskInstance,
	 * ITaskInstance)
	 */
	@Override
	public boolean areSemanticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.SEMANTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSyntacticallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.SYNTACTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSyntacticallyEqual(ITaskInstance,
	 * ITaskInstance)
	 */
	@Override
	public boolean areSyntacticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.SYNTACTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
	}

	/**
	 * <p>
	 * used to to call the task equality rule manager for the comparison of the
	 * two provided children. If no required equality level is provided, than
	 * the most concrete equality is returned. Otherwise, the required equality
	 * is returned as long as the children are equal on that level.
	 * </p>
	 * 
	 * @param child1
	 *            the first task to be compared
	 * @param child2
	 *            the second task to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality
	 */
	private TaskEquality callRuleManager(ITask child1, ITask child2,
			TaskEquality requiredEqualityLevel) {
		if (requiredEqualityLevel == null) {
			return TaskEqualityRuleManager.getInstance()
					.compare(child1, child2);
		} else if (TaskEqualityRuleManager.getInstance().areAtLeastEqual(
				child1, child2, requiredEqualityLevel)) {
			return requiredEqualityLevel;
		} else {
			return TaskEquality.UNEQUAL;
		}
	}

	/**
	 * <p>
	 * used to to call the task equality rule manager for the comparison of the
	 * two provided children. If no required equality level is provided, than
	 * the most concrete equality is returned. Otherwise, the required equality
	 * is returned as long as the children are equal on that level.
	 * </p>
	 * 
	 * @param taskInstance1
	 *            the first task instance to be compared
	 * @param taskInstance2
	 *            the second task instance to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality
	 */
	private TaskEquality callRuleManager(ITaskInstance taskInstance1,
			ITaskInstance taskInstance2, TaskEquality requiredEqualityLevel) {
		if (requiredEqualityLevel == null) {
			return TaskEqualityRuleManager.getInstance().compare(taskInstance1,
					taskInstance2);
		} else if (TaskEqualityRuleManager.getInstance().areAtLeastEqual(
				taskInstance1, taskInstance2, requiredEqualityLevel)) {
			return requiredEqualityLevel;
		} else {
			return TaskEquality.UNEQUAL;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#compare(ITask, ITask)
	 */
	@Override
	public TaskEquality compare(ITask task1, ITask task2) {
		return getEquality(task1, task2, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#compare(ITaskInstance, ITaskInstance)
	 */
	@Override
	public TaskEquality compare(ITaskInstance instance1, ITaskInstance instance2) {
		return getEquality(instance1, instance2, null);
	}

	/**
	 * <p>
	 * compares two sequences with each other checking for the provided required
	 * level of equality. If this level is ensured, the method immediately
	 * returns. The more concrete the required equality level, the more checks
	 * this method performs.
	 * </p>
	 * 
	 * @param task1
	 *            the first task to be compared
	 * @param task2
	 *            the second task to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality.
	 */
	private TaskEquality getEquality(ITask task1, ITask task2,
			TaskEquality requiredEqualityLevel) {
		final List<ITask> children1 = ((ISequence) task1).getChildren();
		final List<ITask> children2 = ((ISequence) task2).getChildren();

		// if both sequences do not have children, they are equal although this
		// doesn't make sense
		if ((children1.size() == 0) && (children2.size() == 0)) {
			return TaskEquality.LEXICALLY_EQUAL;
		}

		if (children1.size() != children2.size()) {
			return TaskEquality.UNEQUAL;
		}

		TaskEquality resultingEquality = TaskEquality.LEXICALLY_EQUAL;
		for (int i = 0; i < children1.size(); i++) {
			final ITask child1 = children1.get(i);
			final ITask child2 = children2.get(i);

			final TaskEquality taskEquality = callRuleManager(child1, child2,
					requiredEqualityLevel);

			if ((taskEquality == null)
					|| (taskEquality == TaskEquality.UNEQUAL)) {
				return TaskEquality.UNEQUAL;
			}

			resultingEquality = resultingEquality
					.getCommonDenominator(taskEquality);
		}

		return resultingEquality;
	}

	/**
	 * <p>
	 * compares two sequence instances with each other checking for the provided
	 * required level of equality. If this level is ensured, the method
	 * immediately returns. The more concrete the required equality level, the
	 * more checks this method performs.
	 * </p>
	 * 
	 * @param taskInstance1
	 *            the first task instance to be compared
	 * @param taskInstance2
	 *            the second task instance to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality.
	 */
	private TaskEquality getEquality(ITaskInstance taskInstance1,
			ITaskInstance taskInstance2, TaskEquality requiredEqualityLevel) {
		final ISequenceInstance sequence1 = (ISequenceInstance) taskInstance1;
		final ISequenceInstance sequence2 = (ISequenceInstance) taskInstance2;

		// if both sequences do not have children, they are equal although this
		// doesn't make sense
		if ((sequence1.size() == 0) && (sequence2.size() == 0)) {
			return TaskEquality.LEXICALLY_EQUAL;
		}

		if (sequence1.size() != sequence2.size()) {
			return TaskEquality.UNEQUAL;
		}

		TaskEquality resultingEquality = TaskEquality.LEXICALLY_EQUAL;
		for (int i = 0; i < sequence1.size(); i++) {
			final ITaskInstance child1 = sequence1.get(i);
			final ITaskInstance child2 = sequence2.get(i);

			final TaskEquality taskEquality = callRuleManager(child1, child2,
					requiredEqualityLevel);

			if ((taskEquality == null)
					|| (taskEquality == TaskEquality.UNEQUAL)) {
				return TaskEquality.UNEQUAL;
			}

			resultingEquality = resultingEquality
					.getCommonDenominator(taskEquality);
		}

		return resultingEquality;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#isApplicable(ITask, ITask)
	 */
	@Override
	public boolean isApplicable(ITask task1, ITask task2) {
		return (task1 instanceof ISequence) && (task2 instanceof ISequence);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#isApplicable(ITaskInstance, ITaskInstance)
	 */
	@Override
	public boolean isApplicable(ITaskInstance instance1, ITaskInstance instance2) {
		return isApplicable(instance1.getTask(), instance2.getTask());
	}
}
