//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.uml;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.emf.common.util.EList;
import org.eclipse.uml2.uml.Activity;
import org.eclipse.uml2.uml.ActivityEdge;
import org.eclipse.uml2.uml.ActivityNode;
import org.eclipse.uml2.uml.CallOperationAction;
import org.eclipse.uml2.uml.Comment;
import org.eclipse.uml2.uml.Component;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interaction;
import org.eclipse.uml2.uml.InteractionFragment;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Lifeline;
import org.eclipse.uml2.uml.Message;
import org.eclipse.uml2.uml.MessageOccurrenceSpecification;
import org.eclipse.uml2.uml.MessageSort;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.Vertex;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SOAPEventType;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SimpleSOAPEventType;
import de.ugoe.cs.autoquest.plugin.uml.eventcore.UMLTransitionType;
import de.ugoe.cs.autoquest.usageprofiles.IStochasticProcess;

/**
 * <p>
 * Utilities for working with UML.
 * </p>
 * 
 * @author Steffen Herbold
 */
public class UMLUtils {

    /**
     * <p>
     * Creates a sequence of events with {@link UMLTransitionType} as event type from a given
     * sequence of events with the {@link SOAPEventType}, by matching the sequences to a state
     * machine.
     * </p>
     * 
     * @param sequence
     *            SOAP sequences
     * @param stateMachine
     *            the state machine
     * @return create UML sequences
     */
    public static List<Event> createUMLTransitionSequence(List<Event> sequence,
                                                          StateMachine stateMachine)
    {
        List<List<Transition>> matchingSequences =
            determineMatchingTransitionSequences(sequence, stateMachine);

        if (matchingSequences.size() != 1) {
            throw new RuntimeException("no unique match found; " + matchingSequences.size() +
                " matches");
        }
        List<Event> umlEventSequence = new LinkedList<>();
        for (Transition transition : matchingSequences.get(0)) {
            umlEventSequence.add(new Event(new UMLTransitionType(transition)));
        }
        return umlEventSequence;
    }

    /**
     * <p>
     * Uses a sequences of events with the {@link UMLTransitionType} to determine the transition
     * probabilities for the state machine.
     * </p>
     * 
     * @param sequences
     *            UML sequences
     * @param stateMachine
     *            state machine to be converted to a usage profile
     */
    public static void convertStateMachineToUsageProfile(Collection<List<Event>> sequences,
                                                         StateMachine stateMachine)
    {
        // create state->outgoings hashmap
        Map<Vertex, Map<Transition, Integer>> stateMap = new HashMap<>();
        for (Region region : stateMachine.getRegions()) {
            for (Vertex state : region.getSubvertices()) {
                stateMap.put(state, new HashMap<Transition, Integer>());
            }
        }

        // create counters for each transition
        for (List<Event> sequence : sequences) {
            for (Event event : sequence) {
                if (event.getType() instanceof UMLTransitionType) {
                    Transition transition = ((UMLTransitionType) event.getType()).getTransition();
                    Map<Transition, Integer> transitionMap = stateMap.get(transition.getSource());
                    Integer value = transitionMap.get(transition);
                    if (value == null) {
                        value = 0;
                    }
                    transitionMap.put(transition, value + 1);
                }
                else {
                    throw new RuntimeException(
                                               "Wrong event type. Only UMLTransitionType supported but was: " +
                                                   event.getType().getClass().getName());
                }
            }
        }

        // calculate probabilities
        for (Region region : stateMachine.getRegions()) {
            for (Vertex state : region.getSubvertices()) {
                Map<Transition, Integer> transitionMap = stateMap.get(state);
                int totalCount = 0;
                for (Entry<Transition, Integer> entry : transitionMap.entrySet()) {
                    totalCount += entry.getValue();
                }
                if (totalCount != 0) {
                    for (Transition transition : state.getOutgoings()) {
                        double prob = 0.0d;
                        if (transitionMap.containsKey(transition)) {
                            prob = ((double) transitionMap.get(transition)) / totalCount;
                        }
                        Comment comment = transition.createOwnedComment();
                        comment.setBody("" + prob);
                    }
                }
                else {
                    // system has never been in this state, all transitions equally likely
                    int numOutgoings = state.getOutgoings().size();
                    for (Transition transition : state.getOutgoings()) {
                        Comment comment = transition.createOwnedComment();
                        comment.setBody("" + (1.0d / numOutgoings));
                    }
                }
            }
        }
    }

    /**
     * <p>
     * Determines all matching {@link Transition} sequences in a state machine for a given sequence
     * of SOAP events.
     * </p>
     * 
     * @param sequence
     *            SOAP sequence
     * @param stateMachine
     *            the state machine
     * @return all matching {@link Transition} sequences
     */
    public static List<List<Transition>> determineMatchingTransitionSequences(List<Event> sequence,
                                                                              StateMachine stateMachine)
    {
        EList<Region> regions = stateMachine.getRegions();
        EList<Vertex> states = null;
        for (Region region : regions) {
            if (states == null) {
                states = region.getSubvertices();
            }
            else {
                states.addAll(region.getSubvertices());
            }
        }
        List<Transition> allTransitions = new LinkedList<>();
        for (Vertex state : states) {
            allTransitions.addAll(state.getOutgoings());
        }

        List<List<Transition>> matchingSequences = null;
        List<Transition> currentTransitions = null;

        // first, we try to find a single unique transition that we can match using the method name
        for (Iterator<Event> eventIterator = sequence.iterator(); eventIterator.hasNext();) {
            Event event = eventIterator.next();
            if (event.getType() instanceof SOAPEventType) {
                SOAPEventType eventType = (SOAPEventType) event.getType();
                if (matchingSequences == null) {
                    matchingSequences = new LinkedList<>();
                    List<Transition> initialMatches = matchTransitions(allTransitions, eventType);
                    for (Transition transition : initialMatches) {
                        List<Transition> candidate = new LinkedList<>();
                        candidate.add(transition);
                        matchingSequences.add(candidate);
                    }
                    currentTransitions = initialMatches;
                }
                else {
                    List<List<Transition>> nextMatchingSequences = new LinkedList<>();
                    List<Transition> nextCurrentTransitions = new LinkedList<>();
                    Iterator<Transition> currentTransitionIterator = currentTransitions.iterator();
                    Iterator<List<Transition>> currentMatchingSequencesIterator =
                        matchingSequences.iterator();
                    while (currentTransitionIterator.hasNext()) {
                        Transition currentTransition = currentTransitionIterator.next();
                        List<Transition> currentMatch = currentMatchingSequencesIterator.next();

                        List<Transition> matches =
                            matchTransitions(currentTransition.getTarget().getOutgoings(),
                                             eventType);
                        if (matches.isEmpty()) {
                            throw new RuntimeException("no matches found");
                        }
                        for (Transition matchingTransition : matches) {
                            List<Transition> candidate = new LinkedList<>(currentMatch);
                            candidate.add(matchingTransition);
                            nextMatchingSequences.add(candidate);
                            nextCurrentTransitions.add(matchingTransition);
                        }
                    }
                    matchingSequences = nextMatchingSequences;
                    currentTransitions = nextCurrentTransitions;
                }
            }
            else {
                throw new RuntimeException(
                                           "Wrong event type. Only UMLTransitionType supported but was: " +
                                               event.getType().getClass().getName());
            }
        }
        return matchingSequences;
    }

    /**
     * <p>
     * Extends a given model with an interaction that represents an observed sequence.
     * </p>
     * 
     * @param sequence
     *            sequence that is added as sequence diagram
     * @param model
     *            UML model to which the interaction is added
     * @param interactionName
     *            name of the interaction
     */
    public static void createInteractionFromEventSequence(List<Event> sequence,
                                                          Model model,
                                                          String interactionName)
    {

        Component testContext = fetchTestContext(model);

        final Profile utpProfile = model.getAppliedProfile("utp");
        final Stereotype utpTestCase = (Stereotype) utpProfile.getOwnedMember("TestCase");
        final Stereotype utpTestComponent = (Stereotype) utpProfile.getOwnedMember("TestComponent");
        final Stereotype utpSUT = (Stereotype) utpProfile.getOwnedMember("SUT");
        
        Operation operation = testContext.createOwnedOperation(interactionName, null, null);
        operation.applyStereotype(utpTestCase);

        Interaction interaction =
            (Interaction) testContext.createPackagedElement(interactionName + "_Impl",
                                                            UMLPackage.Literals.INTERACTION);
        operation.getMethods().add(interaction);

        // create lifelines
        Lifeline userLifeline = null;
        // List<Port> userPorts = new LinkedList<>();
        for (Property property : testContext.getAllAttributes()) {
            if (property.getAppliedStereotypes().contains(utpSUT)) {
                String serviceName = property.getName();
                Lifeline targetLifeline = interaction.createLifeline(serviceName);
                targetLifeline.setRepresents(property);
            }
            else if (property.getType().getAppliedStereotypes().contains(utpTestComponent)) {
                userLifeline = interaction.createLifeline(property.getName());
                userLifeline.setRepresents(property);
            }    
        }
        
        // TODO sanity checks for userLifeline!=null, etc.

        int i = 0;
        for (Event event : sequence) {
            if (!(event.equals(Event.STARTEVENT) || event.equals(Event.ENDEVENT))) {
                String serviceName = getServiceNameFromEvent(event);
                String methodName = getCalledMethodFromEvent(event);
                
                // determine lifelines
                Lifeline msgTargetLifeline;
                Lifeline msgSourceLifeline;
                
                if( serviceName.equals(userLifeline.getName()) ) {
                    // message being send to user
                    // currently we just select the first lifeline that is not the user
                    // this, obviously, has to be replaced with the real service. 
                    // however, identification of the source of a message is still an open issue
                    msgSourceLifeline = null;
                    for( Lifeline lifeline : interaction.getLifelines() ) {
                        if(!lifeline.equals(userLifeline)){
                            msgSourceLifeline = lifeline;
                            break;
                        }
                    }
                    msgTargetLifeline = userLifeline;
                } else {
                    msgSourceLifeline = userLifeline;
                    msgTargetLifeline = interaction.getLifeline(serviceName);
                }

                // determine target interface
                Interface targetInterface = getRealizedInterfaceFromProperty((Property) msgTargetLifeline.getRepresents());
                
                // create message
                MessageOccurrenceSpecification sendFragment =
                    (MessageOccurrenceSpecification) interaction
                        .createFragment(i + ":" + methodName + "_sendFragment",
                                        UMLPackage.Literals.MESSAGE_OCCURRENCE_SPECIFICATION);
                MessageOccurrenceSpecification recvFragment =
                    (MessageOccurrenceSpecification) interaction
                        .createFragment(i + ":" + methodName + "_recvFragment",
                                        UMLPackage.Literals.MESSAGE_OCCURRENCE_SPECIFICATION);

                sendFragment.setCovered(msgSourceLifeline);
                recvFragment.setCovered(msgTargetLifeline);

                Message message = interaction.createMessage(methodName);
                if (getOperationFromName(targetInterface.getOperations(), methodName) == null) {
                    System.out.println("operation not found in the " + targetInterface.getName() + " interface: " + methodName);
                }
                message.setSignature(getOperationFromName(targetInterface.getOperations(),
                                                          methodName));
                message.setMessageSort(MessageSort.ASYNCH_CALL_LITERAL);
                message.setSendEvent(sendFragment);
                message.setReceiveEvent(recvFragment);

                // now the connector needs to be determined
                EList<Property> userAttributes = ((Component) msgSourceLifeline.getRepresents().getType()).getAttributes();
                EList<Property> targetAttributes = ((Component) msgTargetLifeline.getRepresents().getType()).getAttributes();
                
                for( Property userAttribute : userAttributes ) {
                    if( userAttribute instanceof Port ) {
                        EList<ConnectorEnd> userEnds = ((Port) userAttribute).getEnds();
                        for( ConnectorEnd userEnd : userEnds ) {
                            Connector userConnector = (Connector) userEnd.eContainer();
                            for( Property targetAttribute : targetAttributes ) {
                                if( targetAttribute instanceof Port ) {
                                    EList<ConnectorEnd> targetEnds = ((Port) targetAttribute).getEnds();
                                    for( ConnectorEnd targetEnd : targetEnds ) {
                                        Connector targetConnector = (Connector) targetEnd.eContainer();
                                        if( targetConnector==userConnector ) {
                                            message.setConnector(targetConnector);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                
                sendFragment.setMessage(message);
                recvFragment.setMessage(message);

                i++;
            }
        }
    }

    /**
     * <p>
     * Calculates the usage score of an interaction as the logsum of the event probabilities
     * multiplied with the length of the interaction.
     * </p>
     * 
     * @param interaction
     *            interaction for which the score is calculated
     * @param usageProfile
     *            usage profile used for the calculation
     * @return calculated usage score
     */
    public static double calculateUsageScore(Interaction interaction,
                                             IStochasticProcess usageProfile)
    {
        double usageScore = 0.0d;

        EList<InteractionFragment> interactionFragments = interaction.getFragments();
        List<Event> eventSequence = new LinkedList<>();
        eventSequence.add(Event.STARTEVENT);
        for (InteractionFragment interactionFragment : interactionFragments) {
            if (interactionFragment.getName() != null &&
                interactionFragment.getName().endsWith("_recvFragment"))
            {
                String serviceName =
                    interactionFragment.getCovereds().get(0).getRepresents().getName().split("_")[0];
                String methodName = "UNKNOWN";
                if (interactionFragment instanceof MessageOccurrenceSpecification) {
                    methodName =
                        ((MessageOccurrenceSpecification) interactionFragment).getMessage()
                            .getName();
                }
                eventSequence.add(new Event(new SimpleSOAPEventType(methodName, serviceName)));
            }
        }
        eventSequence.add(Event.ENDEVENT);
        double prob = usageProfile.getLogSum(eventSequence);
        usageScore = eventSequence.size() * prob;

        return usageScore;
    }

    /**
     * <p>
     * Extends the given model with an activity for usage-based scheduling of the test cases.
     * </p>
     * 
     * @param model
     *            model to be extended
     * @param usageProfile
     *            usage profile used as foundation
     */
    public static void createScheduling(Model model, IStochasticProcess usageProfile) {

        final Profile utpProfile = model.getAppliedProfile("utp");
        final Stereotype utpTestCase = (Stereotype) utpProfile.getOwnedMember("TestCase");

        Component testContext = fetchTestContext(model);

        Map<Operation, Double> usageScoreMapUnsorted = new HashMap<>();

        // first, we determine all test cases and calculate their usage scores
        for (Operation operation : testContext.getAllOperations()) {
            if (operation.getAppliedStereotypes().contains(utpTestCase)) {
                Interaction interaction = (Interaction) operation.getMethods().get(0);
                usageScoreMapUnsorted
                    .put(operation, calculateUsageScore(interaction, usageProfile));
            }
        }
        Map<Operation, Double> usageScoreMapSorted = sortByValue(usageScoreMapUnsorted);

        // now we create the scheduling
        Activity schedulingActivity =
            (Activity) testContext.createOwnedBehavior("UsageBasedScheduling",
                                                       UMLPackage.Literals.ACTIVITY);
        testContext.setClassifierBehavior(schedulingActivity);

        ActivityNode startNode =
            schedulingActivity.createOwnedNode("final", UMLPackage.Literals.INITIAL_NODE);
        ActivityNode finalNode =
            schedulingActivity.createOwnedNode("final", UMLPackage.Literals.ACTIVITY_FINAL_NODE);

        ActivityNode currentOperationNode = startNode;

        for (Entry<Operation, Double> entry : usageScoreMapSorted.entrySet()) {
            Operation operation = entry.getKey();
            CallOperationAction nextOperationNode =
                (CallOperationAction) schedulingActivity
                    .createOwnedNode(operation.getName(), UMLPackage.Literals.CALL_OPERATION_ACTION);
            nextOperationNode.setOperation(operation);

            ActivityEdge edge =
                schedulingActivity.createEdge(currentOperationNode.getName() + "_to_" +
                    nextOperationNode.getName(), UMLPackage.Literals.CONTROL_FLOW);
            edge.setSource(currentOperationNode);
            edge.setTarget(nextOperationNode);

            currentOperationNode = nextOperationNode;
        }

        ActivityEdge edge =
            schedulingActivity
                .createEdge(currentOperationNode.getName() + "_to_" + finalNode.getName(),
                            UMLPackage.Literals.CONTROL_FLOW);
        edge.setSource(currentOperationNode);
        edge.setTarget(finalNode);
    }

    /**
     * From
     * http://stackoverflow.com/questions/109383/how-to-sort-a-mapkey-value-on-the-values-in-java
     * and adapted to do an inverse sorting
     */
    public static <K, V extends Comparable<? super V>> Map<K, V> sortByValue(Map<K, V> map) {
        List<Map.Entry<K, V>> list = new LinkedList<>(map.entrySet());
        Collections.sort(list, new Comparator<Map.Entry<K, V>>() {
            @Override
            public int compare(Map.Entry<K, V> o1, Map.Entry<K, V> o2) {
                return -1 * (o1.getValue()).compareTo(o2.getValue());
            }
        });

        Map<K, V> result = new LinkedHashMap<>();
        for (Map.Entry<K, V> entry : list) {
            result.put(entry.getKey(), entry.getValue());
        }
        return result;
    }

    /**
     * <p>
     * Helper function to get the name of a service from a SOAP event.
     * </p>
     * 
     * @param event
     *            event for which the service name is retrieved
     * @return service name
     */
    protected static String getServiceNameFromEvent(Event event) {
        if (event.getType() instanceof SOAPEventType) {
            return ((SOAPEventType) event.getType()).getServiceName();
        }
        else if (event.getType() instanceof SimpleSOAPEventType) {
            return ((SimpleSOAPEventType) event.getType()).getServiceName();
        }
        else {
            throw new RuntimeException(
                                       "Wrong event type. Only SOAPEventType and SimpleSOAPEventType supported but was: " +
                                           event.getType().getClass().getName());
        }
    }

    /**
     * 
     * <p>
     * Helper function to get the called method from a SOAP event
     * </p>
     * 
     * @param event
     *            event for which the called method is retrieved
     * @return called method
     */
    private static String getCalledMethodFromEvent(Event event) {
        if (event.getType() instanceof SOAPEventType) {
            return ((SOAPEventType) event.getType()).getCalledMethod();
        }
        else if (event.getType() instanceof SimpleSOAPEventType) {
            return ((SimpleSOAPEventType) event.getType()).getCalledMethod();
        }
        else {
            throw new RuntimeException(
                                       "Wrong event type. Only SOAPEventType and SimpleSOAPEventType supported but was: " +
                                           event.getType().getClass().getName());
        }
    }

    /**
     * <p>
     * Fetches an operation using only its name from a list of operations. Returns the first match
     * found or null if no match is found.
     * </p>
     * 
     * @param operations
     *            list of operations
     * @param name
     *            name of the operation
     * @return first matching operation; null if no match is found
     */
    private static Operation getOperationFromName(EList<Operation> operations, String name) {
        if (name == null) {
            throw new IllegalArgumentException("name of the operation must not be null");
        }
        if (operations != null) {
            for (Operation operation : operations) {
                if (operation.getName() != null && operation.getName().equals(name)) {
                    return operation;
                }
            }
        }
        return null;
    }

    /**
     * <p>
     * Determines which transitions match a given {@link SOAPEventType}.
     * </p>
     * 
     * @param transitions
     *            the transitions
     * @param eventType
     *            the SOAP event
     * @return matching transitions
     */
    private static List<Transition> matchTransitions(List<Transition> transitions,
                                                     SOAPEventType eventType)
    {
        List<Transition> matching = new LinkedList<>();
        for (Transition transition : transitions) {
            // String serviceName = transition.getName().split("\\.")[0]; // TODO service name check
            String methodName = transition.getName().split("\\.")[1];
            if (methodName.equals(eventType.getCalledMethod())) {
                matching.add(transition);
            }
        }
        return matching;
    }
    
    private static Interface getRealizedInterfaceFromProperty(Property property) {
        return getRealizedInterfaceFromComponent((Component) property.getType());
    }
    
    private static Interface getRealizedInterfaceFromComponent(Component comp) {
        Interface myInterface = null;
        for( Property property : comp.getAttributes() ) {
            if( property instanceof Port ) {
                Port port = (Port) property;
                if( !port.isConjugated() ) {
                    if( myInterface==null ) {
                        myInterface = port.getProvideds().get(0);
                    } 
                    else if( myInterface!=port.getProvideds().get(0)) {
                        System.err.println("multiple different interfaces found");
                    }
                }
            }
        }
        return myInterface;
        //return ((Port) comp.getAttributes().get(0)).getInterface();
        //Realization realization = (Realization) comp.getNestedClassifiers().get(0).getRelationships(UMLPackage.Literals.REALIZATION).get(0);
        //return (Interface) realization.getSuppliers().get(0);
    }
    
    private static Component fetchTestContext(Model model) {
        final Profile utpProfile = model.getAppliedProfile("utp");
        final Stereotype utpTestContext = (Stereotype) utpProfile.getOwnedMember("TestContext");
        
        for( Element element : model.getOwnedElements() ) {
            if( element instanceof Component && element.getApplicableStereotypes().contains(utpTestContext) ) {
                return (Component) element;
            }
        }
        return null;
    }
}
