//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.temporalrelation;

import org.junit.Test;

/**
 * @author Patrick Harms
 */
public class CondenseSimilarTasksRuleTest extends AbstractTemporalRelationshipTC {

    /**
     *
     */
    @Test
    public void test_SelectionDetection_01() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Event selAction1 {}" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Event selAction2 {}" +
            "    Event action3 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence3 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction1 {}" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction2 {}" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SelectionDetection_02() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Event selAction1 {}" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Event selAction2 {}" +
            "    Event action3 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence3 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction1 {}" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction2 {}" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SelectionDetection_03() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration iter1 {" +
            "      Sequence sequence2 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Event selAction1 {}" +
            "        Event action2 {}" +
            "      }" +
            "      Sequence sequence2 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Event selAction1 {}" +
            "        Event action2 {}" +
            "      }" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Iteration iter2 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Event selAction2 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence4 {" +
            "    Iteration iter3 {" +
            "      Sequence sequence5 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Selection sel1 {" +
            "          Event selAction1 {}" +
            "        }" +
            "        Event action2 {}" +
            "      }" +
            "      Sequence sequence5 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Selection sel1 {" +
            "          Event selAction1 {}" +
            "        }" +
            "        Event action2 {}" +
            "      }" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Iteration iter3 {" +
            "      Sequence sequence5 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "          Event action1 {}" +
            "        }" +
            "        Selection sel1 {" +
            "          Event selAction2 {}" +
            "        }" +
            "        Event action2 {}" +
            "      }" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
           "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SelectionDetection_04() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Sequence sequence2 {" +
            "      Iteration iter1 {" +
            "        Event action1 {}" +
            "        Event action1 {}" +
            "      }" +
            "      Event selAction1 {}" +
            "      Event action2 {}" +
            "    }" +
            "    Sequence sequence3 {" +
            "      Sequence sequence4 {" +
            "        Iteration iter1 {" +
            "          Event action1 {}" +
            "        }" +
            "      Event selAction2 {}" +
            "      }" +
            "      Event action2 {}" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence5 {" +
            "    Sequence sequence6 {" +
            "      Iteration iter1 {" +
            "        Event action1 {}" +
            "        Event action1 {}" +
            "      }" +
            "      Selection sel1 {" +
            "        Event selAction1 {}" +
            "      }" +
            "      Event action2 {}" +
            "    }" +
            "    Sequence sequence6 {" +
            "      Iteration iter1 {" +
            "        Event action1 {}" +
            "      }" +
            "      Selection sel1 {" +
            "        Event selAction2 {}" +
            "      }" +
            "      Event action2 {}" +
            "    }" +
            "  }" +
           "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_OptionalDetection_01() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Event optAction1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_OptionalDetection_02() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event optAction1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
           "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_OptionalDetection_03() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Event optAction1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence4 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Iteration iter1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
           "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_OptionalDetection_04() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration iter1 {" +
            "      Sequence sequence2 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Event action2 {}" +
            "      }" +
            "      Sequence sequence2 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Event action2 {}" +
            "      }" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Iteration iter2 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Event optAction1 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence4 {" +
            "    Iteration iter3 {" +
            "      Sequence sequence5 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Optional opt1 {}" +
            "        Event action2 {}" +
            "      }" +
            "      Sequence sequence5 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "        }" +
            "        Optional opt1 {}" +
            "        Event action2 {}" +
            "      }" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Iteration iter3 {" +
            "      Sequence sequence5 {" +
            "        Iteration iter2 {" +
            "          Event action1 {}" +
            "          Event action1 {}" +
            "        }" +
            "        Optional opt1 {" +
            "          Event optAction1 {}" +
            "        }" +
            "        Event action2 {}" +
            "      }" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
           "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_OptionalDetection_05() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Sequence sequence2 {" +
            "      Iteration iter1 {" +
            "        Event action1 {}" +
            "        Event action1 {}" +
            "      }" +
            "      Event action2 {}" +
            "    }" +
            "    Sequence sequence3 {" +
            "      Sequence sequence4 {" +
            "        Iteration iter1 {" +
            "          Event action1 {}" +
            "        }" +
            "        Event optAction1 {}" +
            "      }" +
            "      Event action2 {}" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence5 {" +
            "    Sequence sequence6 {" +
            "      Iteration iter1 {" +
            "        Event action1 {}" +
            "        Event action1 {}" +
            "      }" +
            "      Optional opt1 {}" +
            "      Event action2 {}" +
            "    }" +
            "    Sequence sequence6 {" +
            "      Iteration iter1 {" +
            "        Event action1 {}" +
            "      }" +
            "      Optional opt1 {" +
            "        Event optAction1 {}" +
            "      }" +
            "      Event action2 {}" +
            "    }" +
            "  }" +
           "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_OptionalDetection_06() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Iteration it1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Iteration it1 {" +
            "        Event optAction1 {}" +
            "      }" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_OptionalDetection_07() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Optional op1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_IterationPreservation_01() throws Exception {
        String input =
            "UserSession {" +
            "  Iteration itSequence1 {" +
            "    Sequence sequence1 {" +
            "      Event action1 {}" +
            "      Event optAction1 {}" +
            "      Event action2 {}" +
            "    }" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Iteration itSequence3 {" +
            "    Sequence sequence3 {" +
            "      Event action1 {}" +
            "      Optional opt1 {" +
            "        Event optAction1 {}" +
            "      }" +
            "      Event action2 {}" +
            "    }" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_IterationPreservation_02() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration itSequence2 {" +
            "      Sequence sequence2 {" +
            "        Event action1 {}" +
            "        Event action2 {}" +
            "      }" +
            "      Sequence sequence2 {" +
            "        Event action1 {}" +
            "        Event action2 {}" +
            "      }" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Iteration itSequence4 {" +
            "      Sequence sequence4 {" +
            "        Event action4 {}" +
            "        Event action3 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Selection sel1 {" +
            "    Sequence sequence1 {" +
            "      Iteration itSequence2 {" +
            "        Sequence sequence2 {" +
            "          Event action1 {}" +
            "          Event action2 {}" +
            "        }" +
            "        Sequence sequence2 {" +
            "          Event action1 {}" +
            "          Event action2 {}" +
            "        }" +
            "      }" +
            "      Event action3 {}" +
            "    }" +
            "  }" +
            "  Selection sel1 {" +
            "    Sequence sequence3 {" +
            "      Event action1 {}" +
            "      Iteration itSequence4 {" +
            "        Sequence sequence4 {" +
            "          Event action4 {}" +
            "          Event action3 {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_IterationPreservation_03() throws Exception {
        String input =
            "UserSession {" +
            "  Iteration itSequence1 {" +
            "    Sequence sequence1 {" +
            "      Iteration itSequence2 {" +
            "        Sequence sequence2 {" +
            "          Event action1 {}" +
            "          Event action2 {}" +
            "        }" +
            "        Sequence sequence2 {" +
            "          Event action1 {}" +
            "          Event action2 {}" +
            "        }" +
            "      }" +
            "      Event action3 {}" +
            "    }" +
            "  }" +
            "  Iteration itSequence3 {" +
            "    Sequence sequence3 {" +
            "      Event action1 {}" +
            "      Iteration itSequence4 {" +
            "        Sequence sequence4 {" +
            "          Event action4 {}" +
            "          Event action3 {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Iteration itSelection1 {" +
            "    Selection sel1 {" +
            "      Sequence sequence1 {" +
            "        Iteration itSequence2 {" +
            "          Sequence sequence2 {" +
            "            Event action1 {}" +
            "            Event action2 {}" +
            "          }" +
            "          Sequence sequence2 {" +
            "            Event action1 {}" +
            "            Event action2 {}" +
            "          }" +
            "        }" +
            "        Event action3 {}" +
            "      }" +
            "    }" +
            "    Selection sel1 {" +
            "      Sequence sequence3 {" +
            "        Event action1 {}" +
            "        Iteration itSequence4 {" +
            "          Sequence sequence4 {" +
            "            Event action4 {}" +
            "            Event action3 {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_IterationPreservation_04() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration itSequence2 {" +
            "      Sequence sequence2 {" +
            "        Event action1 {}" +
            "        Iteration itSequence3 {" +
            "          Sequence sequence3 {" +
            "            Event action2 {}" +
            "            Event action3 {}" +
            "          }" +
            "          Sequence sequence3 {" +
            "            Event action2 {}" +
            "            Event action3 {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Event action4 {}" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Event action1 {}" +
            "    Iteration itSequence5 {" +
            "      Sequence sequence5 {" +
            "        Iteration itSequence6 {" +
            "          Sequence sequence6 {" +
            "            Event action2 {}" +
            "            Event action5 {}" +
            "          }" +
            "        }" +
            "        Event action4 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Selection sel1 {" +
            "    Sequence sequence1 {" +
            "      Iteration itSequence2 {" +
            "        Sequence sequence2 {" +
            "          Event action1 {}" +
            "          Iteration itSequence3 {" +
            "            Sequence sequence3 {" +
            "              Event action2 {}" +
            "              Event action3 {}" +
            "            }" +
            "            Sequence sequence3 {" +
            "              Event action2 {}" +
            "              Event action3 {}" +
            "            }" +
            "          }" +
            "        }" +
            "      }" +
            "      Event action4 {}" +
            "    }" +
            "  }" +
            "  Selection sel1 {" +
            "    Sequence sequence4 {" +
            "      Event action1 {}" +
            "      Iteration itSequence5 {" +
            "        Sequence sequence5 {" +
            "          Iteration itSequence6 {" +
            "            Sequence sequence6 {" +
            "              Event action2 {}" +
            "              Event action5 {}" +
            "            }" +
            "          }" +
            "          Event action4 {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_IterationPreservation_05() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event noiseAction1 {}" +
            "    Iteration itSequence2 {" +
            "      Sequence sequence2 {" +
            "        Iteration itSequence3 {" +
            "          Sequence sequence3 {" +
            "            Event action1 {}" +
            "            Iteration itSequence4 {" +
            "              Sequence sequence4 {" +
            "                Event action2 {}" +
            "                Event action3 {}" +
            "              }" +
            "            }" +
            "          }" +
            "        }" +
            "        Event action4 {}" +
            "      }" +
            "      Sequence sequence2 {" +
            "        Iteration itSequence3 {" +
            "          Sequence sequence3 {" +
            "            Event action1 {}" +
            "            Iteration itSequence4 {" +
            "              Sequence sequence4 {" +
            "                Event action2 {}" +
            "                Event action3 {}" +
            "              }" +
            "            }" +
            "          }" +
            "        }" +
            "        Event action4 {}" +
            "      }" +
            "    }" +
            "    Event noiseAction2 {}" +
            "  }" +
            "  Sequence sequence5 {" +
            "    Event noiseAction1 {}" +
            "    Iteration itSequence6 {" +
            "      Sequence sequence6 {" +
            "        Event action1 {}" +
            "        Iteration itSequence7 {" +
            "          Sequence sequence7 {" +
            "            Iteration itSequence8 {" +
            "              Sequence sequence8 {" +
            "                Event action5 {}" +
            "                Event action3 {}" +
            "              }" +
            "            }" +
            "            Event action4 {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Event noiseAction2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence9 {" +
            "    Event noiseAction1 {}" +
            "    Iteration itSelection1 {" +
            "      Selection sel1 {" +
            "        Sequence sequence2 {" +
            "          Iteration itSequence3 {" +
            "            Sequence sequence3 {" +
            "              Event action1 {}" +
            "              Iteration itSequence4 {" +
            "                Sequence sequence4 {" +
            "                  Event action2 {}" +
            "                  Event action3 {}" +
            "                }" +
            "              }" +
            "            }" +
            "          }" +
            "          Event action4 {}" +
            "        }" +
            "      }" +
            "      Selection sel1 {" +
            "        Sequence sequence2 {" +
            "          Iteration itSequence3 {" +
            "            Sequence sequence3 {" +
            "              Event action1 {}" +
            "              Iteration itSequence4 {" +
            "                Sequence sequence4 {" +
            "                  Event action2 {}" +
            "                  Event action3 {}" +
            "                }" +
            "              }" +
            "            }" +
            "          }" +
            "          Event action4 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Event noiseAction2 {}" +
            "  }" +
            "  Sequence sequence9 {" +
            "    Event noiseAction1 {}" +
            "    Iteration itSelection1 {" +
            "      Selection sel1 {" +
            "        Sequence sequence6 {" +
            "          Event action1 {}" +
            "          Iteration itSequence7 {" +
            "            Sequence sequence7 {" +
            "              Iteration itSequence8 {" +
            "                Sequence sequence8 {" +
            "                  Event action5 {}" +
            "                  Event action3 {}" +
            "                }" +
            "              }" +
            "              Event action4 {}" +
            "            }" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Event noiseAction2 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_IterationPreservation_06() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration itSequence2 {" +
            "      Sequence sequence2 {" +
            "        Event action1 {}" +
            "        Event action2 {}" +
            "        Event action3 {}" +
            "      }" +
            "    }" +
            "    Iteration itSequence3 {" +
            "      Sequence sequence3 {" +
            "        Event action4 {}" +
            "        Event action5 {}" +
            "        Event action6 {}" +
            "      }" +
            "      Sequence sequence3 {" +
            "        Event action4 {}" +
            "        Event action5 {}" +
            "        Event action6 {}" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Event action5 {}" +
            "    Event action6 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Selection sel1 {" +
            "    Sequence sequence1 {" +
            "      Iteration itSequence2 {" +
            "        Sequence sequence2 {" +
            "          Event action1 {}" +
            "          Event action2 {}" +
            "          Event action3 {}" +
            "        }" +
            "      }" +
            "      Iteration itSequence3 {" +
            "        Sequence sequence3 {" +
            "          Event action4 {}" +
            "          Event action5 {}" +
            "          Event action6 {}" +
            "        }" +
            "        Sequence sequence3 {" +
            "          Event action4 {}" +
            "          Event action5 {}" +
            "          Event action6 {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "  Selection sel1 {" +
            "    Sequence sequence4 {" +
            "      Event action1 {}" +
            "      Event action2 {}" +
            "      Event action5 {}" +
            "      Event action6 {}" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     * TODO: correct to let it run through
     */
    /*@Test
    public void test_IterationPreservation_07() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Sequence sequence2 {" +
            "      Event action1 {}" +
            "      Event action2 {}" +
            "    }" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Sequence sequence4 {" +
            "      Sequence sequence2 {" +
            "        Event action1 {}" +
            "        Event action2 {}" +
            "      }" +
            "      Iteration it1 {" +
            "        Event inserted1 {}" +
            "      }" +
            "    }" +
            "    Iteration it2 {" +
            "      Sequence sequence5 {" +
            "        Event inserted2 {}" +
            "        Event action3 {}" +
            "      }" +
            "      Sequence sequence5 {" +
            "        Event inserted2 {}" +
            "        Event action3 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Selection sel1 {" +
            "    Sequence sequence1 {" +
            "      Sequence sequence2 {" +
            "        Event action1 {}" +
            "        Event action2 {}" +
            "      }" +
            "      Event action3 {}" +
            "    }" +
            "  }" +
            "  Selection sel1 {" +
            "    Sequence sequence3 {" +
            "      Sequence sequence4 {" +
            "        Sequence sequence2 {" +
            "          Event action1 {}" +
            "          Event action2 {}" +
            "        }" +
            "        Iteration it1 {" +
            "          Event inserted1 {}" +
            "        }" +
            "      }" +
            "      Iteration it2 {" +
            "        Sequence sequence5 {" +
            "          Event inserted2 {}" +
            "          Event action3 {}" +
            "        }" +
            "        Sequence sequence5 {" +
            "          Event inserted2 {}" +
            "          Event action3 {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_NoSimilarityDetection_01() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event selAction1 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event selAction2 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event selAction1 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event selAction2 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_NoSimilarityDetection_02() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Event selAction1 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Event selAction2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Event selAction1 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Event selAction2 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_UniquenessOfReplacements_01() throws Exception {
        // situation where the same optional task is identified for two different sets of similar
        // tasks. However, the replacements must have the same optional task as child although
        // the parents are different.
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Event optAction1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action3 {}" +
            "    Event optAction1 {}" +
            "    Event action4 {}" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Event action3 {}" +
            "    Event action4 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence5 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence5 {" +
            "    Event action1 {}" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence6 {" +
            "    Event action3 {}" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action4 {}" +
            "  }" +
            "  Sequence sequence6 {" +
            "    Event action3 {}" +
            "    Optional opt1 {}" +
            "    Event action4 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_UniquenessOfReplacements_02() throws Exception {
        // situation where the same selection task is identified for two different sets of similar
        // tasks. However, the replacements must have the same selection task as child although
        // the parents are different.
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Event selAction1 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Event selAction2 {}" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Iteration it2 {" +
            "      Event action3 {}" +
            "      Event action3 {}" +
            "    }" +
            "    Event selAction1 {}" +
            "    Event action4 {}" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Iteration it2 {" +
            "      Event action3 {}" +
            "    }" +
            "    Event selAction2 {}" +
            "    Event action4 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence5 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence5 {" +
            "    Iteration it1 {" +
            "      Event action1 {}" +
            "      Event action1 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction2 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence6 {" +
            "    Iteration it2 {" +
            "      Event action3 {}" +
            "      Event action3 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction1 {}" +
            "    }" +
            "    Event action4 {}" +
            "  }" +
            "  Sequence sequence6 {" +
            "    Iteration it2 {" +
            "      Event action3 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event selAction2 {}" +
            "    }" +
            "    Event action4 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SpecificIntermediateSituations_01() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Optional op1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence2 {" +
            "    Event action1 {}" +
            "    Event optAction1 {}" +
            "    Event action2 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Event optAction1 {}" +
            "    }" +
            "    Event action2 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SpecificIntermediateSituations_02() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Event action1 {}" +
            "    Iteration itSeq2 {" +
            "      Sequence sequence2 {" +
            "        Event optAction1 {}" +
            "        Event optAction2 {}" +
            "      }" +
            "      Sequence sequence2 {" +
            "        Event optAction1 {}" +
            "        Event optAction2 {}" +
            "      }" +
            "    }" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "    Event action4 {}" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "    Event action4 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence4 {" +
            "    Event action1 {}" +
            "    Optional opt1 {" +
            "      Iteration itSeq2 {" +
            "        Sequence sequence2 {" +
            "          Event optAction1 {}" +
            "          Event optAction2 {}" +
            "        }" +
            "        Sequence sequence2 {" +
            "          Event optAction1 {}" +
            "          Event optAction2 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "    Event action4 {}" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Event action1 {}" +
            "    Optional opt1 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "    Event action4 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SpecificIntermediateSituations_03() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence 6635 {" +
            "    Sequence 6464 {" +
            "      Iteration 6108 {" +
            "        Event action1 {}" +
            "      }" +
            "      Sequence 6305 {" +
            "        Sequence 6284 {" +
            "          Sequence 6198 {" +
            "            Iteration 6111 {" +
            "              Event action2 {}" +
            "            }" +
            "            Event action3 {}" +
            "          }" +
            "          Iteration 6089 {" +
            "            Event action8 {}" +
            "          }" +
            "        }" +
            "        Event action4 {}" +
            "      }" +
            "      Sequence 6211 {" +
            "        Event action5 {}" +
            "        Event action6 {}" +
            "      }" +
            "    }" +
            "    Sequence 6350 {" +
            "      Iteration 6089 {" +
            "        Event action8 {}" +
            "      }" +
            "      Iteration 6189 {" +
            "        Event action7 {}" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence 6381 {" +
            "    Iteration 6108 {" +
            "      Event action1 {}" +
            "    }" +
            "    Sequence 6258 {" +
            "      Sequence 6218 {" +
            "        Sequence 6199 {" +
            "          Sequence 6198 {" +
            "            Iteration 6111 {" +
            "              Event action2 {}" +
            "            }" +
            "            Event action3 {}" +
            "          }" +
            "          Event action4 {}" +
            "        }" +
            "        Sequence 6211 {" +
            "          Event action5 {}" +
            "          Event action6 {}" +
            "        }" +
            "      }" +
            "      Iteration 6189 {" +
            "        Event action7 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence4 {" +
            "    Iteration 6108 {" +
            "      Event action1 {}" +
            "    }" +
            "    Sequence 6198 {" +
            "      Iteration 6111 {" +
            "        Event action2 {}" +
            "      }" +
            "      Event action3 {}" +
            "    }" +
            "    Optional opt1 {" +
            "      Iteration 6089 {" +
            "        Event action8 {}" +
            "      }" +
            "    }" +
            "    Event action4 {}" +
            "    Sequence 6211 {" +
            "      Event action5 {}" +
            "      Event action6 {}" +
            "    }" +
            "    Optional opt1 {" +
            "      Iteration 6089 {" +
            "        Event action8 {}" +
            "      }" +
            "    }" +
            "    Iteration 6189 {" +
            "      Event action7 {}" +
            "    }" +
            "  }" +
            "  Sequence sequence4 {" +
            "    Iteration 6108 {" +
            "      Event action1 {}" +
            "    }" +
            "    Sequence 6198 {" +
            "      Iteration 6111 {" +
            "        Event action2 {}" +
            "      }" +
            "      Event action3 {}" +
            "    }" +
            "    Optional opt1 { }" +
            "    Event action4 {}" +
            "    Sequence 6211 {" +
            "      Event action5 {}" +
            "      Event action6 {}" +
            "    }" +
            "    Optional opt1 { }" +
            "    Iteration 6189 {" +
            "      Event action7 {}" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SpecificIntermediateSituations_04() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Event action1 {}" +
            "    Sequence seq2 {" +
            "      Event action2 {}" +
            "      Optional 221221 {" +
            "        Event action9 {}" +
            "      }" +
            "      Event action4 {}" +
            "      Optional 221221 {" +
            "        Event action9 {}" +
            "      }" +
            "      Event action5 {}" +
            "    }" +
            "    Sequence seq3 {" +
            "      Sequence seq4 {" +
            "        Event action6 {}" +
            "        Event action7 {}" +
            "      }" +
            "      Event action9 {}" +
            "    }" +
            "  }" +
            "  Sequence seq5 {" +
            "    Iteration 211092 {" +
            "      Sequence seq6 {" +
            "        Event action1 {}" +
            "        Sequence seq7 {" +
            "          Sequence seq8 {" +
            "            Event action2 {}" +
            "            Optional 221296 {" +
            "              Event action3 {}" +
            "            }" +
            "            Event action4 {}" +
            "            Event action5 {}" +
            "          }" +
            "          Sequence seq4 {" +
            "            Event action6 {}" +
            "            Event action7 {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration #210928 {" +
            "      Sequence seq9 {" +
            "        Event action8 {}" +
            "        Event action9 {}" +
            "      }" +
            "      Sequence seq9 {" +
            "        Event action8 {}" +
            "        Event action9 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence seq10 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Selection 24 {" +
            "      Optional 4 {" +
            "        Event action9{}" +
            "      }" +
            "    }" +
            "    Event action5{}" +
            "    Optional 4 {" +
            "      Event action9{}" +
            "    }" +
            "    Event action5{}" +
            "    Sequence seq4 {" +
            "      Event action6 {}" +
            "      Event action7 {}" +
            "    }" +
            "    Iteration 27 {" +
            "      Sequence seq12 {" +
            "        Optional 25 {" +
            "        }" +
            "        Event action9{}" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence seq10 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Selection 24 {" +
            "      Optional 14 {" +
            "        Event action3 {}" +
            "      }" +
            "    }" +
            "    Event action5{}" +
            "    Optional 4 {" +
            "    }" +
            "    Event action5{}" +
            "    Sequence seq4 {" +
            "      Event action6 {}" +
            "      Event action7 {}" +
            "    }" +
            "    Iteration 27 {" +
            "      Sequence seq12 {" +
            "        Optional 25 {" +
            "          Event action8 {}" +
            "        }" +
            "        Event action9{}" +
            "      }" +
            "      Sequence seq12 {" +
            "        Optional 25 {" +
            "          Event action8 {}" +
            "        }" +
            "        Event action9{}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     * TODO: correct to let it run through
     */
    /*@Test
    public void test_SpecificIntermediateSituations_05() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence 1 {" +
            "    Iteration 211092 {" +
            "      Sequence 2 {" +
            "        Iteration 210420 {" +
            "          Event action1 {}" +
            "        }" +
            "        Sequence 3 {" +
            "          Sequence 4 {" +
            "            Iteration 210339 {" +
            "              Event action2 {}" +
            "            }" +
            "            Optional 221296 {" +
            "              Iteration 210776 {" +
            "                Event action3 {}" +
            "              }" +
            "            }" +
            "            Iteration 210452 {" +
            "              Event action4 {}" +
            "            }" +
            "            Iteration 210453 {" +
            "              Event action5 {}" +
            "            }" +
            "          }" +
            "          Sequence 5 {" +
            "            Iteration 210422 {" +
            "              Event action6 {}" +
            "            }" +
            "            Iteration 210456 {" +
            "              Event action7 {}" +
            "            }" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration #210928 {" +
            "      Sequence 6 {" +
            "        Iteration 210369 {" +
            "          Event action8 {}" +
            "        }" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "      Sequence 6 {" +
            "        Iteration 210369 {" +
            "          Event action8 {}" +
            "        }" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence 7 {" +
            "    Iteration 210420 {" +
            "      Event action1 {}" +
            "    }" +
            "    Sequence 8 {" +
            "      Sequence 9 {" +
            "        Iteration 210339 {" +
            "          Event action2 {}" +
            "        }" +
            "      }" +
            "      Optional 221221 {" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "      Iteration 210452 {" +
            "        Event action4 {}" +
            "      }" +
            "      Optional 221221 {" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "      Iteration 210453 {" +
            "        Event action5 {}" +
            "      }" +
            "    }" +
            "    Sequence 10 {" +
            "      Sequence 11 {" +
            "        Iteration 210422 {" +
            "          Event action6 {}" +
            "        }" +
            "        Iteration 210456 {" +
            "          Event action7 {}" +
            "        }" +
            "      }" +
            "      Iteration 210337 {" +
            "        Event action9 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence12 {" +
            "    Iteration 211092 {" +
            "      Sequence 2 {" +
            "        Iteration 210420 {" +
            "          Event action1 {}" +
            "        }" +
            "        Sequence 3 {" +
            "          Sequence 4 {" +
            "            Iteration 210339 {" +
            "              Event action2 {}" +
            "            }" +
            "            Optional 221296 {" +
            "              Iteration 210776 {" +
            "                Event action3 {}" +
            "              }" +
            "            }" +
            "            Iteration 210452 {" +
            "              Event action4 {}" +
            "            }" +
            "            Iteration 210453 {" +
            "              Event action5 {}" +
            "            }" +
            "          }" +
            "          Sequence 5 {" +
            "            Iteration 210422 {" +
            "              Event action6 {}" +
            "            }" +
            "            Iteration 210456 {" +
            "              Event action7 {}" +
            "            }" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration #210928 {" +
            "      Sequence 6 {" +
            "        Iteration 210369 {" +
            "          Event action8 {}" +
            "        }" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "      Sequence 6 {" +
            "        Iteration 210369 {" +
            "          Event action8 {}" +
            "        }" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence 7 {" +
            "    Iteration 210420 {" +
            "      Event action1 {}" +
            "    }" +
            "    Sequence 8 {" +
            "      Sequence 9 {" +
            "        Iteration 210339 {" +
            "          Event action2 {}" +
            "        }" +
            "      }" +
            "      Optional 221221 {" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "      Iteration 210452 {" +
            "        Event action4 {}" +
            "      }" +
            "      Optional 221221 {" +
            "        Iteration 210337 {" +
            "          Event action9 {}" +
            "        }" +
            "      }" +
            "      Iteration 210453 {" +
            "        Event action5 {}" +
            "      }" +
            "    }" +
            "    Sequence 10 {" +
            "      Sequence 11 {" +
            "        Iteration 210422 {" +
            "          Event action6 {}" +
            "        }" +
            "        Iteration 210456 {" +
            "          Event action7 {}" +
            "        }" +
            "      }" +
            "      Iteration 210337 {" +
            "        Event action9 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     * TODO: correct to let it run through
     */
    /*@Test
    public void test_SpecificIntermediateSituations_06() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Event action3 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Selection sel3 {" +
            "      Event action3 {}" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence seq3 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Selection sel3 {" +
            "      Event action3 {}" +
            "    }" +
            "  }" +
            "  Sequence seq3 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Selection sel3 {" +
            "      Event action3 {}" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SpecificIntermediateSituations_07() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Event action1 {}" +
            "    Iteration it1 {" +
            "      Event action2 {}" +
            "      Event action2 {}" +
            "    }" +
            "    Selection sel1 {" +
            "      Event action3 {}" +
            "    }" +
            "    Event action5 {}" +
            "  }" +
            "  Sequence seq2 {" +
            "    Event action1 {}" +
            "    Iteration it1 {" +
            "      Event action2 {}" +
            "    }" +
            "    Selection sel2 {" +
            "      Event action4 {}" +
            "    }" +
            "    Event action5 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence seq3 {" +
            "    Event action1 {}" +
            "    Iteration it1 {" +
            "      Event action2 {}" +
            "      Event action2 {}" +
            "    }" +
            "    Selection sel3 {" +
            "      Event action3 {}" +
            "    }" +
            "    Event action5 {}" +
            "  }" +
            "  Sequence seq3 {" +
            "    Event action1 {}" +
            "    Iteration it1 {" +
            "      Event action2 {}" +
            "    }" +
            "    Selection sel3 {" +
            "      Event action4 {}" +
            "    }" +
            "    Event action5 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     * TODO: correct to let it run through
     */
    /*@Test
    public void test_SpecificIntermediateSituations_08() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Optional opt2 {" +
            "      Sequence seq2 {" +
            "        Event action5 {}" +
            "        Event action6 {}" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Optional opt2 {}" +
            "  }" +
            "  Sequence seq3 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Optional opt1 {" +
            "      Event action4 {}" +
            "    }" +
            "    Sequence seq2 {" +
            "      Event action5 {}" +
            "      Event action6 {}" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence seq4 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Optional opt1 {}" +
            "    Optional opt2 {" +
            "      Sequence seq2 {" +
            "        Event action5 {}" +
            "        Event action6 {}" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence seq4 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Optional opt1 {}" +
            "    Optional opt2 {}" +
            "  }" +
            "  Sequence seq4 {" +
            "    Event action1 {}" +
            "    Event action2 {}" +
            "    Optional opt1 {" +
            "      Event action4 {}" +
            "    }" +
            "    Optional opt2 {" +
            "      Sequence seq2 {" +
            "        Event action5 {}" +
            "        Event action6 {}" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     *
     */
    @Test
    public void test_SpecificIntermediateSituations_09() throws Exception {
        String input =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Sequence seq2 {" +
            "      Event action0 {}" +
            "      Event action1 {}" +
            "      Event action2 {}" +
            "      Event action3 {}" +
            "    }" +
            "    Event action4 {}" +
            "    Event action5 {}" +
            "  }" +
            "  Sequence seq3 {" +
            "    Sequence seq4 {" +
            "      Event action0 {}" +
            "      Event action1 {}" +
            "      Event action3 {}" +
            "    }" +
            "    Event action4 {}" +
            "    Event action5 {}" +
            "  }" +
            "  Sequence seq5 {" +
            "    Sequence seq2 {" +
            "      Event action0 {}" +
            "      Event action1 {}" +
            "      Event action2 {}" +
            "      Event action3 {}" +
            "    }" +
            "    Event action5 {}" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence seq6 {" +
            "    Sequence seq7 {" +
            "      Event action0 {}" +
            "      Event action1 {}" +
            "      Optional opt1 {" +
            "        Event action2 {}" +
            "      }" +
            "      Event action3 {}" +
            "    }" +
            "    Optional opt2 {" +
            "      Event action4 {}" +
            "    }" +
            "    Event action5 {}" +
            "  }" +
            "  Sequence seq6 {" +
            "    Sequence seq7 {" +
            "      Event action0 {}" +
            "      Event action1 {}" +
            "      Optional opt1 {}" +
            "      Event action3 {}" +
            "    }" +
            "    Optional opt2 {" +
            "      Event action4 {}" +
            "    }" +
            "    Event action5 {}" +
            "  }" +
            "  Sequence seq6 {" +
            "    Sequence seq7 {" +
            "      Event action0 {}" +
            "      Event action1 {}" +
            "      Optional opt1 {" +
            "        Event action2 {}" +
            "      }" +
            "      Event action3 {}" +
            "    }" +
            "    Optional opt2 {}" +
            "    Event action5 {}" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     * TODO: correct to let it run through
     */
    /*@Test
    public void test_RealLifeScenario_01() throws Exception {
        // situation where the same selection task is identified for two different sets of similar
        // tasks. However, the replacements must have the same selection task as child although
        // the parents are different.
        String input =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Iteration itSeq2 {" +
            "      Sequence seq2 {" +
            "        Iteration itSeq3 {" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "        }" +
            "        Iteration itEnterTf2 {" +
            "          Event enterTf2 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration itSeqConfirm {" +
            "      Sequence seqConfirm {" +
            "        Iteration itClickBtn {" +
            "          Event clickBtn {}" +
            "        }" +
            "        Iteration itScroll {" +
            "          Event scroll {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence seq5 {" +
            "    Iteration itSeq4 {" +
            "      Sequence seq4 {" +
            "        Iteration itClickTf1 {" +
            "          Event clickTf1 {}" +
            "        }" +
            "        Iteration itEnterTf1 {" +
            "          Event enterTf1 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Sequence seq6 {" +
            "      Iteration itSeq7 {" +
            "        Sequence seq7 {" +
            "          Iteration itClickTf2 {" +
            "            Event clickTf2 {}" +
            "          }" +
            "          Iteration itEnterTf2 {" +
            "            Event enterTf2 {}" +
            "          }" +
            "        }" +
            "      }" +
            "      Iteration itSeqConfirm {" +
            "        Sequence seqConfirm {" +
            "          Iteration itClickBtn {" +
            "            Event clickBtn {}" +
            "          }" +
            "          Iteration itScroll {" +
            "            Event scroll {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Selection sel1 {" +
            "    Sequence seq8 {" +
            "      Iteration itSeq2 {" +
            "        Sequence seq2 {" +
            "          Iteration itSeq3 {" +
            "            Sequence seq3 {" +
            "              Iteration itSeq4 {" +
            "                Sequence seq4 {" +
            "                  Iteration itClickTf1 {" +
            "                    Event clickTf1 {}" +
            "                  }" +
            "                  Iteration itEnterTf1 {" +
            "                    Event enterTf1 {}" +
            "                  }" +
            "                }" +
            "              }" +
            "              Event tabPress {}" +
            "            }" +
            "          }" +
            "          Iteration itEnterTf2 {" +
            "            Event enterTf2 {}" +
            "          }" +
            "        }" +
            "      }" +
            "      Iteration itSeqConfirm {" +
            "        Sequence seqConfirm {" +
            "          Iteration itClickBtn {" +
            "            Event clickBtn {}" +
            "          }" +
            "          Iteration itScroll {" +
            "            Event scroll {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "  Selection sel1 {" +
            "    Sequence seq9 {" +
            "      Iteration itSeq4 {" +
            "        Sequence seq4 {" +
            "          Iteration itClickTf1 {" +
            "            Event clickTf1 {}" +
            "          }" +
            "          Iteration itEnterTf1 {" +
            "            Event enterTf1 {}" +
            "          }" +
            "        }" +
            "      }" +
            "      Sequence seq6 {" +
            "        Iteration itSeq7 {" +
            "          Sequence seq7 {" +
            "            Iteration itClickTf2 {" +
            "              Event clickTf2 {}" +
            "            }" +
            "            Iteration itEnterTf2 {" +
            "              Event enterTf2 {}" +
            "            }" +
            "          }" +
            "        }" +
            "        Iteration itSeqConfirm {" +
            "          Sequence seqConfirm {" +
            "            Iteration itClickBtn {" +
            "              Event clickBtn {}" +
            "            }" +
            "            Iteration itScroll {" +
            "              Event scroll {}" +
            "            }" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     * TODO: correct to let it run through
     */
    /*@Test
    public void test_RealLifeScenario_02() throws Exception {
        // situation where the same selection task is identified for two different sets of similar
        // tasks. However, the replacements must have the same selection task as child although
        // the parents are different.
        String input =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Iteration itSeq2 {" +
            "      Sequence seq2 {" +
            "        Iteration itSeq3 {" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "        }" +
            "        Iteration itEnterTf2 {" +
            "          Event enterTf2 {}" +
            "        }" +
            "      }" +
            "      Sequence seq2 {" +
            "        Iteration itSeq3 {" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "        }" +
            "        Iteration itEnterTf2 {" +
            "          Event enterTf2 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration itSeqConfirm {" +
            "      Sequence seqConfirm {" +
            "        Iteration itClickBtn {" +
            "          Event clickBtn {}" +
            "        }" +
            "        Iteration itScroll {" +
            "          Event scroll {}" +
            "        }" +
            "      }" +
            "      Sequence seqConfirm {" +
            "        Iteration itClickBtn {" +
            "          Event clickBtn {}" +
            "        }" +
            "        Iteration itScroll {" +
            "          Event scroll {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence seq5 {" +
            "    Iteration itSeq4 {" +
            "      Sequence seq4 {" +
            "        Iteration itClickTf1 {" +
            "          Event clickTf1 {}" +
            "        }" +
            "        Iteration itEnterTf1 {" +
            "          Event enterTf1 {}" +
            "        }" +
            "      }" +
            "      Sequence seq4 {" +
            "        Iteration itClickTf1 {" +
            "          Event clickTf1 {}" +
            "        }" +
            "        Iteration itEnterTf1 {" +
            "          Event enterTf1 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Sequence seq6 {" +
            "      Iteration itSeq7 {" +
            "        Sequence seq7 {" +
            "          Iteration itClickTf2 {" +
            "            Event clickTf2 {}" +
            "          }" +
            "          Iteration itEnterTf2 {" +
            "            Event enterTf2 {}" +
            "          }" +
            "        }" +
            "        Sequence seq7 {" +
            "          Iteration itClickTf2 {" +
            "            Event clickTf2 {}" +
            "          }" +
            "          Iteration itEnterTf2 {" +
            "            Event enterTf2 {}" +
            "          }" +
            "        }" +
            "      }" +
            "      Iteration itSeqConfirm {" +
            "        Sequence seqConfirm {" +
            "          Iteration itClickBtn {" +
            "            Event clickBtn {}" +
            "          }" +
            "          Iteration itScroll {" +
            "            Event scroll {}" +
            "          }" +
            "        }" +
            "        Sequence seqConfirm {" +
            "          Iteration itClickBtn {" +
            "            Event clickBtn {}" +
            "          }" +
            "          Iteration itScroll {" +
            "            Event scroll {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence seq8 {" +
            "    Iteration itSeq2 {" +
            "      Sequence seq2 {" +
            "        Iteration itSeq3 {" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "        }" +
            "        Iteration itEnterTf2 {" +
            "          Event enterTf2 {}" +
            "        }" +
            "      }" +
            "      Sequence seq2 {" +
            "        Iteration itSeq3 {" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "          Sequence seq3 {" +
            "            Iteration itSeq4 {" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "              Sequence seq4 {" +
            "                Iteration itClickTf1 {" +
            "                  Event clickTf1 {}" +
            "                }" +
            "                Iteration itEnterTf1 {" +
            "                  Event enterTf1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event tabPress {}" +
            "          }" +
            "        }" +
            "        Iteration itEnterTf2 {" +
            "          Event enterTf2 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration itSeqConfirm {" +
            "      Sequence seqConfirm {" +
            "        Iteration itClickBtn {" +
            "          Event clickBtn {}" +
            "        }" +
            "        Iteration itScroll {" +
            "          Event scroll {}" +
            "        }" +
            "      }" +
            "      Sequence seqConfirm {" +
            "        Iteration itClickBtn {" +
            "          Event clickBtn {}" +
            "        }" +
            "        Iteration itScroll {" +
            "          Event scroll {}" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence seq5 {" +
            "    Iteration itSeq4 {" +
            "      Sequence seq4 {" +
            "        Iteration itClickTf1 {" +
            "          Event clickTf1 {}" +
            "        }" +
            "        Iteration itEnterTf1 {" +
            "          Event enterTf1 {}" +
            "        }" +
            "      }" +
            "      Sequence seq4 {" +
            "        Iteration itClickTf1 {" +
            "          Event clickTf1 {}" +
            "        }" +
            "        Iteration itEnterTf1 {" +
            "          Event enterTf1 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Sequence seq6 {" +
            "      Iteration itSeq7 {" +
            "        Sequence seq7 {" +
            "          Iteration itClickTf2 {" +
            "            Event clickTf2 {}" +
            "          }" +
            "          Iteration itEnterTf2 {" +
            "            Event enterTf2 {}" +
            "          }" +
            "        }" +
            "        Sequence seq7 {" +
            "          Iteration itClickTf2 {" +
            "            Event clickTf2 {}" +
            "          }" +
            "          Iteration itEnterTf2 {" +
            "            Event enterTf2 {}" +
            "          }" +
            "        }" +
            "      }" +
            "      Iteration itSeqConfirm {" +
            "        Sequence seqConfirm {" +
            "          Iteration itClickBtn {" +
            "            Event clickBtn {}" +
            "          }" +
            "          Iteration itScroll {" +
            "            Event scroll {}" +
            "          }" +
            "        }" +
            "        Sequence seqConfirm {" +
            "          Iteration itClickBtn {" +
            "            Event clickBtn {}" +
            "          }" +
            "          Iteration itScroll {" +
            "            Event scroll {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }

    /**
     * TODO: correct to let it run through
     */
    /*@Test
    public void test_RealLifeScenario_03() throws Exception {
        // situation where the same selection task is identified for two different sets of similar
        // tasks. However, the replacements must have the same selection task as child although
        // the parents are different.
        String input =
            "UserSession {" +
            "  Sequence sequence1 {" +
            "    Iteration itSeqUtilizeTextField1 {" +
            "      Sequence seqUtilizeTextField1 {" +
            "        Iteration itClickTextField1 {" +
            "          Event clickTextField1 {}" +
            "          Event clickTextField1 {}" +
            "        }" +
            "        Iteration itEnterTextTextField1 {" +
            "          Event enterTextTextField1 {}" +
            "        }" +
            "      }" +
            "      Sequence seqUtilizeTextField1 {" +
            "        Iteration itClickTextField1 {" +
            "          Event clickTextField1 {}" +
            "        }" +
            "        Iteration itEnterTextTextField1 {" +
            "          Event enterTextTextField1 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Sequence sequence2 {" +
            "      Iteration itSeqUtilizeTextField2 {" +
            "        Sequence seqUtilizeTextField2 {" +
            "          Iteration itClickTextField2 {" +
            "            Event clickTextField2 {}" +
            "          }" +
            "          Iteration itEnterTextTextField2 {" +
            "            Event enterTextTextField2 {}" +
            "            Event enterTextTextField2 {}" +
            "          }" +
            "        }" +
            "        Sequence seqUtilizeTextField2 {" +
            "          Iteration itClickTextField2 {" +
            "            Event clickTextField2 {}" +
            "            Event clickTextField2 {}" +
            "          }" +
            "          Iteration itEnterTextTextField2 {" +
            "            Event enterTextTextField2 {}" +
            "          }" +
            "        }" +
            "      }" +
            "      Iteration itClickButton1 {" +
            "        Event clickButton1 {}" +
            "        Event clickButton1 {}" +
            "      }" +
            "    }" +
            "  }" +
            "  Sequence sequence3 {" +
            "    Iteration itSequence4 {" +
            "      Sequence sequence4 {" +
            "        Iteration itSequence5 {" +
            "          Sequence sequence5 {" +
            "            Iteration itSeqUtilizeTextField1 {" +
            "              Sequence seqUtilizeTextField1 {" +
            "                Iteration itClickTextField1 {" +
            "                  Event clickTextField1 {}" +
            "                }" +
            "                Iteration itEnterTextTextField1 {" +
            "                  Event enterTextTextField1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event pressTabTextField2 {}" +
            "          }" +
            "          Sequence sequence5 {" +
            "            Iteration itSeqUtilizeTextField1 {" +
            "              Sequence seqUtilizeTextField1 {" +
            "                Iteration itClickTextField1 {" +
            "                  Event clickTextField1 {}" +
            "                }" +
            "                Iteration itEnterTextTextField1 {" +
            "                  Event enterTextTextField1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event pressTabTextField2 {}" +
            "          }" +
            "        }" +
            "        Iteration itEnterTextTextField2 {" +
            "          Event enterTextTextField2 {}" +
            "          Event enterTextTextField2 {}" +
            "        }" +
            "      }" +
            "      Sequence sequence4 {" +
            "        Iteration itSequence5 {" +
            "          Sequence sequence5 {" +
            "            Iteration itSeqUtilizeTextField1 {" +
            "              Sequence seqUtilizeTextField1 {" +
            "                Iteration itClickTextField1 {" +
            "                  Event clickTextField1 {}" +
            "                }" +
            "                Iteration itEnterTextTextField1 {" +
            "                  Event enterTextTextField1 {}" +
            "                  Event enterTextTextField1 {}" +
            "                }" +
            "              }" +
            "              Sequence seqUtilizeTextField1 {" +
            "                Iteration itClickTextField1 {" +
            "                  Event clickTextField1 {}" +
            "                  Event clickTextField1 {}" +
            "                }" +
            "                Iteration itEnterTextTextField1 {" +
            "                  Event enterTextTextField1 {}" +
            "                }" +
            "              }" +
            "            }" +
            "            Event pressTabTextField2 {}" +
            "          }" +
            "        }" +
            "        Iteration itEnterTextTextField2 {" +
            "          Event enterTextTextField2 {}" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration itClickButton1 {" +
            "      Event clickButton1 {}" +
            "    }" +
            "  }" +
            "}";

        String output =
            "UserSession {" +
            "  Sequence sequence6 {" +
            "    Selection selection1 {" +
            "      Sequence sequence7 {" +
            "        Iteration itSeqUtilizeTextField1 {" +
            "          Sequence seqUtilizeTextField1 {" +
            "            Iteration itClickTextField1 {" +
            "              Event clickTextField1 {}" +
            "              Event clickTextField1 {}" +
            "            }" +
            "            Iteration itEnterTextTextField1 {" +
            "              Event enterTextTextField1 {}" +
            "            }" +
            "          }" +
            "          Sequence seqUtilizeTextField1 {" +
            "            Iteration itClickTextField1 {" +
            "              Event clickTextField1 {}" +
            "            }" +
            "            Iteration itEnterTextTextField1 {" +
            "              Event enterTextTextField1 {}" +
            "            }" +
            "          }" +
            "        }" +
            "        Iteration itSeqUtilizeTextField2 {" +
            "          Sequence seqUtilizeTextField2 {" +
            "            Iteration itClickTextField2 {" +
            "              Event clickTextField2 {}" +
            "            }" +
            "            Iteration itEnterTextTextField2 {" +
            "              Event enterTextTextField2 {}" +
            "              Event enterTextTextField2 {}" +
            "            }" +
            "          }" +
            "          Sequence seqUtilizeTextField2 {" +
            "            Iteration itClickTextField2 {" +
            "              Event clickTextField2 {}" +
            "              Event clickTextField2 {}" +
            "            }" +
            "            Iteration itEnterTextTextField2 {" +
            "              Event enterTextTextField2 {}" +
            "            }" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration itClickButton1 {" +
            "      Event clickButton1 {}" +
            "      Event clickButton1 {}" +
            "    }" +
            "  }" +
            "  Sequence sequence6 {" +
            "    Selection selection1 {" +
            "      Iteration itSeq4 {" +
            "        Sequence sequence4 {" +
            "          Iteration itSequence5 {" +
            "            Sequence sequence5 {" +
            "              Iteration itSeqUtilizeTextField1 {" +
            "                Sequence seqUtilizeTextField1 {" +
            "                  Iteration itClickTextField1 {" +
            "                    Event clickTextField1 {}" +
            "                  }" +
            "                  Iteration itEnterTextTextField1 {" +
            "                    Event enterTextTextField1 {}" +
            "                  }" +
            "                }" +
            "              }" +
            "              Event pressTabTextField2 {}" +
            "            }" +
            "            Sequence sequence5 {" +
            "              Iteration itSeqUtilizeTextField1 {" +
            "                Sequence seqUtilizeTextField1 {" +
            "                  Iteration itClickTextField1 {" +
            "                    Event clickTextField1 {}" +
            "                  }" +
            "                  Iteration itEnterTextTextField1 {" +
            "                    Event enterTextTextField1 {}" +
            "                  }" +
            "                }" +
            "              }" +
            "              Event pressTabTextField2 {}" +
            "            }" +
            "          }" +
            "          Iteration itEnterTextTextField2 {" +
            "            Event enterTextTextField2 {}" +
            "            Event enterTextTextField2 {}" +
            "          }" +
            "        }" +
            "        Sequence sequence4 {" +
            "          Iteration itSequence5 {" +
            "            Sequence sequence5 {" +
            "              Iteration itSeqUtilizeTextField1 {" +
            "                Sequence seqUtilizeTextField1 {" +
            "                  Iteration itClickTextField1 {" +
            "                    Event clickTextField1 {}" +
            "                  }" +
            "                  Iteration itEnterTextTextField1 {" +
            "                    Event enterTextTextField1 {}" +
            "                    Event enterTextTextField1 {}" +
            "                  }" +
            "                }" +
            "                Sequence seqUtilizeTextField1 {" +
            "                  Iteration itClickTextField1 {" +
            "                    Event clickTextField1 {}" +
            "                    Event clickTextField1 {}" +
            "                  }" +
            "                  Iteration itEnterTextTextField1 {" +
            "                    Event enterTextTextField1 {}" +
            "                  }" +
            "                }" +
            "              }" +
            "              Event pressTabTextField2 {}" +
            "            }" +
            "          }" +
            "          Iteration itEnterTextTextField2 {" +
            "            Event enterTextTextField2 {}" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "    Iteration itClickButton1 {" +
            "      Event clickButton1 {}" +
            "    }" +
            "  }" +
            "}";

        applySessionScopeRule(CondenseSimilarTasksRule.class, input, output);
    }*/
}
