//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.guimodel;

import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec;

/**
 * <p>
 * This is a GUI element specification for tags in HTML documents. Each tag belongs to a certain
 * document. However, for similarity comparison, two page elements are similar even if they do
 * not belong to the same document. Each page element has a tag name and either an id or at least
 * an index in the list of siblings of the same type.
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTMLPageElementSpec extends HTMLGUIElementSpec implements IGUIElementSpec {

    /**
     * <p>
     * default serial version UID
     * </p>
     */
    private static final long serialVersionUID = 1L;
    
    /**
     * <p>
     * the page to which the represented tag belongs
     * </p>
     */
    private HTMLDocumentSpec page;
    
    /**
     * <p>
     * the name of the tag represented by this specification
     * </p>
     */
    private String tagName;
    
    /**
     * <p>
     * the id of the tag represented by this specification, i.e., the value of the id attribute
     * of the tag. May be null in the case the id attribute of the tag is not set. 
     * </p>
     */
    private String htmlId;
    
    /**
     * <p>
     * the index of the tag (0 based) in the list of siblings in the same parent being of the
     * same type. If, e.g., a parent has three li tags as children, the first will have index 0,
     * the second index 1 and the third index2. The indexes are ignored, if the tag have an
     * assigned id. 
     * </p>
     */
    private int index;

    /**
     * <p>
     * initializes the specification with the page to which the represented tag belongs, the tags
     * name, its id or its index.
     * </p>
     *
     * @param page    the page to which the represented tag belongs
     * @param tagName the name of the tag represented by this specification
     * @param htmlId  the id of the tag represented by this specification
     * @param index   the index of the tag
     * 
     * @throws IllegalArgumentException if page and name are null or if neither an id nor an index
     *                                  are provided correctly
     */
    public HTMLPageElementSpec(HTMLDocumentSpec page, String tagName, String htmlId, int index) {
        super(tagName, (page != null ? page.hashCode() : 0) +
              (tagName != null ? tagName.hashCode() : 0) + 
              (htmlId != null ? htmlId.hashCode() : 0) + (index >= 0 ? index : 0));
        
        if (page == null) {
            throw new IllegalArgumentException("page must not be null");
        }
        else if (tagName == null) {
            throw new IllegalArgumentException("tag must not be null");
        }
        else if ((htmlId == null) && (index < 0)) {
            throw new IllegalArgumentException
                ("either id must not be null or the index must be greater or equal to 0");
        }
        
        this.page = page;
        this.tagName = tagName;
        this.htmlId = htmlId;
        this.index = index;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec#getSimilarity(IGUIElementSpec)
     */
    @Override
    public boolean getSimilarity(IGUIElementSpec other) {
        if (other instanceof HTMLPageElementSpec) {
            HTMLPageElementSpec otherSpec = (HTMLPageElementSpec) other;
            
            if (!super.getSimilarity(otherSpec)) {
                return false;
            }
            /*else if (!page.getSimilarity(otherSpec.page)) {
                return false;
            }*/
            else if (!tagName.equals(otherSpec.tagName)) {
                return false;
            }
            
            if (htmlId != null) {
                return htmlId.equals(otherSpec.htmlId);
            }
            else if (index >= 0) {
                return index == otherSpec.index;
            }
        }
        
        return false;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        String str = getTagName();
        
        if ((getHtmlId() != null) && (!"".equals(getHtmlId()))) {
            str += "(id=\"" + getHtmlId() + "\")";
        }
        else {
            str += "[" + getIndex() + "]";
        }
        
        return str;
    }

    /**
     * <p>
     * returns the page to which the represented tag belongs
     * </p>
     *
     * @return the page to which the represented tag belongs
     */
    HTMLDocumentSpec getPage() {
        return page;
    }

    /**
     * <p>
     * returns the name of the tag represented by this specification
     * </p>
     *
     * @return the name of the tag represented by this specification
     */
    String getTagName() {
        return tagName;
    }

    /**
     * <p>
     * returns the id of the tag represented by this specification
     * </p>
     *
     * @return the id of the tag represented by this specification
     */
    String getHtmlId() {
        return htmlId;
    }

    /**
     * <p>
     * returns the index of the tag
     * </p>
     *
     * @return the index of the tag
     */
    int getIndex() {
        return index;
    }

}
