//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.commands;

import java.io.File;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.logging.Level;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.plugin.jfc.JFCSimplifiedLogParser;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCGUIElement;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCGUIElementSpec;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCMenu;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCMenuButton;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Steffen Herbold
 */
public class CMDparseJFCDirwithJacaretoIndices implements Command {

    private List<String> menuList;

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String path;
        String sequencesName = "sequences";

        try {
            path = (String) parameters.get(0);
            if (parameters.size() >= 2) {
                sequencesName = (String) parameters.get(1);
            }
            if (parameters.size() >= 3) {
                menuList =
                    Files.readAllLines(Paths.get((String) parameters.get(2)),
                                       Charset.defaultCharset());
            }

        }
        catch (Exception e) {
            throw new IllegalArgumentException();
        }

        File folder = new File(path);
        if (!folder.isDirectory()) {
            Console.printerrln(path + " is not a directory");
            return;
        }

        JFCSimplifiedLogParser parser = new JFCSimplifiedLogParser();

        String absolutPath = folder.getAbsolutePath();
        for (String filename : folder.list()) {
            String source = absolutPath + File.separator + filename;
            Console.traceln(Level.INFO, "Processing file: " + source);

            try {
                parser.parseFile(source);
            }
            catch (Exception e) {
                Console.printerrln("Could not parse " + source + ": " + e.getMessage());
            }
        }

        Collection<List<Event>> sequences = parser.getSequences();

        GUIModel targets = parser.getGuiModel();

        generateJacaretoIndices(targets.getRootElements(), targets);

        if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
            CommandHelpers.dataOverwritten(sequencesName);
        }

        if (GlobalDataContainer.getInstance().addData(sequencesName + "_targets", targets)) {
            CommandHelpers.dataOverwritten(sequencesName + "_targets");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        // TODO Auto-generated method stub
        System.out.println("parseJFCDirwithJacaretoIndices <path> {<sequencesName>} {<menuFile>}");
        return null;
    }

    // duplicates to parseJFCwithJacaretoIndices
    private int findPopupMenuIndex(IGUIElement item, GUIModel model) {
        // TODO: refactor
        int index = -1;
        List<IGUIElement> children = model.getChildren(item);
        IGUIElement menuChild = null;

        // try to identify this popup menu by looking at its children
        // find the first menu item child of this item
        for (IGUIElement child : children) {
            if (child instanceof JFCMenuButton || child instanceof JFCMenu) {
                menuChild = child;
                break;
            }
        }

        if (menuChild == null) {
            // this popup menu cannot be identified
            // TODO: exception, logging etc
            return -1;
        }

        // find line that contains this menu item name
        String itemName = ((JFCGUIElement) menuChild).getName().trim().toLowerCase();
        int lineOfItem = -1;

        for (int i = 0; i < menuList.size(); i++) {
            String name = "\"" + menuList.get(i).trim().toLowerCase() + "\"";
            if (name.equals(itemName)) {
                lineOfItem = i;
                break;
            }
        }

        if (lineOfItem == -1) {
            // failed to find this item in the menu file
            // TODO: exception, logging etc
            return -1;
        }

        // find menu item's parent line
        String stripped = menuList.get(lineOfItem).replaceFirst("^ *", "");
        int indent = menuList.get(lineOfItem).length() - stripped.length();

        if (indent != 0) {
            for (int i = lineOfItem; i >= 0; i--) {
                stripped = menuList.get(i).replaceFirst("^ *", "");
                int oldIndent = menuList.get(i).length() - stripped.length();

                if (oldIndent < indent) {
                    lineOfItem = i;
                    break;
                }
            }
        }

        // get the item's indentation
        stripped = menuList.get(lineOfItem).replaceFirst("^ *", "");
        indent = menuList.get(lineOfItem).length() - stripped.length();

        if (indent == 0) {
            // top level menu item, just count in which position it is
            for (int i = 0; i <= lineOfItem; i++) {
                if (!menuList.get(i).startsWith(" ")) {
                    index++;
                }
            }
        }
        else {
            // find the topmenu index in the menu file by going backwards
            for (int i = lineOfItem; i >= 0; i--) {
                stripped = menuList.get(i).replaceFirst("^ *", "");
                int oldIndent = menuList.get(i).length() - stripped.length();

                if (oldIndent < indent) {
                    index = lineOfItem - i;
                    break;
                }
            }
        }

        return index;
    }

    private void generateJacaretoIndices(List<IGUIElement> elements, GUIModel targets) {
        HashMap<String, Integer> typeCount = new HashMap<>();

        for (IGUIElement child : elements) {
            String type = child.getSpecification().getType();
            Integer count = typeCount.get(type);

            if (count == null) {
                count = 0;
                typeCount.put(type, count);
            }
            else {
                typeCount.put(type, ++count);
            }

            if (menuList != null && type.equals("javax.swing.JPopupMenu")) {
                // try to use a workaround for popup menu index problems
                int index = findPopupMenuIndex((IGUIElement) child, targets);
                if (index != -1) {
                    ((JFCGUIElementSpec) child.getSpecification()).setAltIndex(index);
                }
                else {
                    // workaround failed, use normal method as fallback
                    ((JFCGUIElementSpec) child.getSpecification()).setAltIndex(count);
                }
            }
            else {
                ((JFCGUIElementSpec) child.getSpecification()).setAltIndex(count);
            }

            generateJacaretoIndices(targets.getChildren(child), targets);
        }
    }

}
