//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.uml;

import java.io.File;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.logging.Level;

import org.apache.commons.lang.SerializationUtils;
import org.eclipse.uml2.uml.Interaction;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.UMLPackage;
import org.junit.After;
import org.junit.BeforeClass;
import org.junit.Test;

import de.fraunhofer.fokus.testing.ModelUtils;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.http.HTTPLogParser;
import de.ugoe.cs.autoquest.plugin.http.HTTPUtils;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SOAPEventType;
import de.ugoe.cs.autoquest.plugin.uml.eventcore.UMLTransitionType;
import de.ugoe.cs.autoquest.testgeneration.RandomWalkGenerator;
import de.ugoe.cs.autoquest.usageprofiles.FirstOrderMarkovModel;
import de.ugoe.cs.autoquest.usageprofiles.IStochasticProcess;
import de.ugoe.cs.util.console.TextConsole;

/**
 * <p>
 * Tests for AutoQUESTs UMLUtils
 * </p>
 * 
 * @author Steffen Herbold
 */
public class UMLUtilsTest {

    private final static String OUTPUT_DIR = "target/tmp/test-outputs/";

    @BeforeClass
    public static void setUpBeforeClass() throws Exception {
        new TextConsole(Level.INFO);
    }
    
    @After
    public void tearDown() throws Exception {
        deleteFiles(new File(OUTPUT_DIR));
    }

    @Test(expected = java.lang.RuntimeException.class)
    public void testCreateUMLTransitionSequence_1() throws Exception {
        // parse log file
        HTTPLogParser parser = new HTTPLogParser();
        parser.parseFile(new File(ClassLoader.getSystemResource("createSequence_1_usagedata.log")
            .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();
        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("createSequence_1_model.uml"));

        StateMachine stateMachine =
            (StateMachine) model.getPackagedElement("PatientIDBehavior", true,
                                                    UMLPackage.Literals.STATE_MACHINE, true);

        Collection<List<Event>> umlSequences = new LinkedList<>();
        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
                else {
                    if (!event.getType().toString().contains("ixsmq")) {
                        eventIter.remove();
                    }
                }
            }

            umlSequences.add(UMLUtils.createUMLTransitionSequence(httpSequence, stateMachine));
        }
    }

    @Test
    public void testConvertStateMachineToUsageProfile_1() throws Exception {
        // parse log file
        HTTPLogParser parser = new HTTPLogParser();
        parser.parseFile(new File(ClassLoader.getSystemResource("createSequence_1_usagedata.log")
            .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();
        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("createSequence_1_model.uml"));

        StateMachine stateMachine =
            (StateMachine) model.getPackagedElement("PatientIDBehavior", true,
                                                    UMLPackage.Literals.STATE_MACHINE, true);

        Collection<List<Event>> umlSequences = new LinkedList<>();
        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
                else {
                    if (!event.getType().toString().contains("ixsmq")) {
                        eventIter.remove();
                    }
                }
            }

            List<List<Transition>> matchingSequences =
                UMLUtils.determineMatchingTransitionSequences(httpSequence, stateMachine);
            if (matchingSequences.size() >= 1) {
                List<Event> umlEventSequence = new LinkedList<>();
                for (Transition transition : matchingSequences.get(0)) {
                    umlEventSequence.add(new Event(new UMLTransitionType(transition)));
                }
                umlSequences.add(umlEventSequence);
            }
        }
        UMLUtils.convertStateMachineToUsageProfile(umlSequences, stateMachine);

        ModelUtils.writeModelToFile(model, OUTPUT_DIR + "convertStateMachineToUsageProfile_1.uml");
    }

    @Test
    public void testCreateInteractionFromEventSequence_1() throws Exception {
        // parse log file
        HTTPLogParser parser =
            new HTTPLogParser(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_1_properties.txt")
                .getFile()));
        parser
            .parseFile(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_1_usagedata.log")
                .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();

        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("testCreateInteractionFromEventSequence_1_model.uml"));

        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
            }

            UMLUtils.createInteractionFromEventSequence(httpSequence, model, "testSequence", null);
        }

        ModelUtils.writeModelToFile(model, OUTPUT_DIR + "testCreateInteractionFromEventSequence_1_result.uml");
    }
    
    @Test
    public void testCreateInteractionFromEventSequence_2() throws Exception {
        // parse log file
        HTTPLogParser parser =
            new HTTPLogParser(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_2_properties.txt")
                .getFile()));
        parser
            .parseFile(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_2_usagedata.log")
                .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();

        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("testCreateInteractionFromEventSequence_2_model.uml"));

        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
            }

            UMLUtils.createInteractionFromEventSequence(httpSequence, model, "testSequence", null);
        }

        ModelUtils.writeModelToFile(model, OUTPUT_DIR + "testCreateInteractionFromEventSequence_1_result.uml");
    }
    
    @Test
    public void testHL7v2_1() throws Exception {
        // parse log file
        HTTPLogParser parser =
            new HTTPLogParser(new File(ClassLoader
                .getSystemResource("hl7_servicenamemap.txt")
                .getFile()));
        parser
            .parseFile(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_1_usagedata.log")
                .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();

        
        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("hl7model_v2.uml"));

        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
            }

            UMLUtils.createInteractionFromEventSequence(httpSequence, model, "testSequence", "RLUSTestSuite_1");
        }

        ModelUtils.writeModelToFile(model, OUTPUT_DIR + "testCreateInteractionFromEventSequence_2_result.uml");
        
    }

    @Test
    public void testCalculateUsageScore_1() throws Exception {
        // parse log file
        HTTPLogParser parser =
            new HTTPLogParser(new File(ClassLoader
                .getSystemResource("testCalculateUsageScore_1_properties.txt").getFile()));
        parser.parseFile(new File(ClassLoader
            .getSystemResource("testCalculateUsageScore_1_usagedata.log").getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();

        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("testCalculateUsageScore_1_model.uml"));

        Collection<List<Event>> simpleSOAPSequences = new LinkedList<>();
        for (List<Event> httpSequence : httpSequences) {
            for (Iterator<Event> eventIter = httpSequence.iterator(); eventIter.hasNext();) {
                Event event = eventIter.next();
                if (!(event.getType() instanceof SOAPEventType)) {
                    eventIter.remove();
                }
            }
            simpleSOAPSequences.add(HTTPUtils.convertToSimpleSOAPEvent(httpSequence));
            // simpleSOAPSequences.add(httpSequence);
        }

        FirstOrderMarkovModel usageProfile = new FirstOrderMarkovModel(new Random(1));
        usageProfile.train(simpleSOAPSequences);

        Collection<List<Event>> genSeqs =
            (new RandomWalkGenerator(10, 1, 100, true, 500)).generateTestSuite(usageProfile);

        int i = 1;
        int[] lengths = new int[genSeqs.size()];
        for (List<Event> seq : genSeqs) {
            UMLUtils.createInteractionFromEventSequence(seq, model, "seq_" + i, null);
            lengths[i - 1] = seq.size();
            i++;
        }
        for (int j = 0; j < genSeqs.size(); j++) {
            Interaction interaction =
                (Interaction) model.getPackagedElement("seq_" + j, true,
                                                       UMLPackage.Literals.INTERACTION, true);
            double usageScore = UMLUtils.calculateUsageScore(interaction, usageProfile);
            System.out.format("usage score %02d: %.2f \t %d\n", j + 1, usageScore, lengths[j]);
        }
    }
    
    @Test
    public void testCreateScheduling_1() throws Exception {
        Model model =
                ModelUtils.loadModel(ClassLoader
                    .getSystemResourceAsStream("testCreateScheduling_1_testsuite.uml"));
        
        IStochasticProcess usageProfile = (IStochasticProcess) SerializationUtils.deserialize(ClassLoader.getSystemResourceAsStream("testCreateScheduling_1_usageprofile.dat"));
        
        UMLUtils.createScheduling(model, usageProfile, null);
        
        ModelUtils.writeModelToFile(model, OUTPUT_DIR + "testCreateScheduling_1_result.uml");
    }
    
    @Test
    public void testValidateModelWithLog_1() throws Exception {
        HTTPLogParser parser =
                new HTTPLogParser(new File(ClassLoader
                    .getSystemResource("testCreateInteractionFromEventSequence_1_properties.txt")
                    .getFile()));
        parser
            .parseFile(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_1_usagedata.log")
                .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();

        
        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("testCreateInteractionFromEventSequence_1_model.uml"));
        
        int violations = UMLUtils.validateModelWithLog(httpSequences, model, null);
        if( violations==0 ) {
            System.out.println("No problems found.");
        } else {
                System.out.println(violations + " violations found.");
        }
    }
    
    @Test
    public void testValidateModelWithLog_2() throws Exception {
        HTTPLogParser parser =
                new HTTPLogParser(new File(ClassLoader
                    .getSystemResource("testCreateInteractionFromEventSequence_2_properties.txt")
                    .getFile()));
        parser
            .parseFile(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_2_usagedata.log")
                .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();

        
        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("testCreateInteractionFromEventSequence_2_model.uml"));
        
        int violations = UMLUtils.validateModelWithLog(httpSequences, model, null);
        if( violations==0 ) {
            System.out.println("No problems found.");
        } else {
            System.out.println(violations + " violations found.");
        }
    }
    
    @Test
    public void testValidateModelWithLog_HL7_v2() throws Exception {
        HTTPLogParser parser =
                new HTTPLogParser(new File(ClassLoader
                    .getSystemResource("hl7_servicenamemap.txt")
                    .getFile()));
        parser
            .parseFile(new File(ClassLoader
                .getSystemResource("testCreateInteractionFromEventSequence_1_usagedata.log")
                .getFile()));
        Collection<List<Event>> httpSequences = parser.getSequences();

        
        Model model =
            ModelUtils.loadModel(ClassLoader
                .getSystemResourceAsStream("hl7model_v2.uml"));
        
        int violations = UMLUtils.validateModelWithLog(httpSequences, model, "IXSTestSuite_1");
        if( violations==0 ) {
            System.out.println("No problems found.");
        } else {
            System.out.println(violations + " violations found.");
        }
    }

    private void deleteFiles(File file) {
        if (file.exists()) {
            if (file.isDirectory()) {
                for (File child : file.listFiles()) {
                    deleteFiles(child);
                }
            }

            try {
                file.delete();
            }
            catch (Exception e) {
                // ignore and delete as much as possible
            }
        }
    }

}
