//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.usability;

import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.tasktrees.manager.TaskTreeManager;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * This command merges similar tasks in a given task tree. It uses the {@link TaskTreeManager} for
 * this purpose. Please consult the documentation of the task tree manager for more details.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDmergeSimilarTasks implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "mergeSimilarTasks <inputTaskTree> {<outputTaskTree>}";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String inputTaskTreeName;
        String outputTaskTreeName;
        try {
            inputTaskTreeName = (String) parameters.get(0);
            if (parameters.size() > 1) {
                outputTaskTreeName = (String) parameters.get(1);
            }
            else {
                outputTaskTreeName = inputTaskTreeName;
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("must provide an input task tree name");
        }

        ITaskModel inputTaskModel = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(inputTaskTreeName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(inputTaskTreeName);
            return;
        }
        if (!(dataObject instanceof ITaskModel)) {
            CommandHelpers.objectNotType(inputTaskTreeName, "ITaskModel");
            return;
        }

        inputTaskModel = (ITaskModel) dataObject;
        
        inputTaskModel = inputTaskModel.clone();
        
        ITaskModel taskModel = new TaskTreeManager().mergeSimilarTasks(inputTaskModel);
        
        if (GlobalDataContainer.getInstance().addData(outputTaskTreeName, taskModel)) {
            CommandHelpers.dataOverwritten(outputTaskTreeName);
        }
        
    }

}
