//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.usability2.rules.patterns;

import java.util.Collection;
import java.util.List;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.eventcore.gui.MouseClick;
import de.ugoe.cs.autoquest.eventcore.gui.MouseInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.TextInput;
import de.ugoe.cs.autoquest.eventcore.gui.ValueSelection;
import de.ugoe.cs.autoquest.usability.EvaluationMethodCaller;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescriptionResolver;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRule;
import de.ugoe.cs.autoquest.usability.rules.UsabilityUsageProblem;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.filter.MultiEventTypeFilter;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.wrapper.EventSequence;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.wrapper.EventSequences;
import de.ugoe.cs.autoquest.plugin.usability2.rules.operator.wrapper.GenerateInstanceListVisitor;
import de.ugoe.cs.autoquest.plugin.usability2.statistics.Histogramm;
import de.ugoe.cs.autoquest.plugin.usability2.tools.TaskUtilities;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIterationInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * <p>
 * Test Rule to see if new pattern method may be used for problem checking
 * </p>
 * 
 * @author Konni Hartmann
 */
public class InputMethodSwitching extends UsabilityRule implements UsabilityUsageProblem {

    /**
     * <p>
     * TODO: comment
     * </p>
     * 
     * @param taskTree
     */
    public InputMethodSwitching(ITaskModel taskModel) {
        super(taskModel);
        this.name = "InputMethodSwitching";
        this.defect = new UsabilityProblemDescriptionResolver().descriptionFor("InputMethodSwitching");
        initUsagePattern();
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     * 
     */
    private void initUsagePattern() {}

    private static enum InputState {
        unknown, mouse, keyboard
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usability.rules.UsabilityRule#check()
     */
    @Override
    public Optional<UsabilityProblemDescription> check() {
        Optional<UsabilityProblemDescription> present = Optional.absent();

        float maxRatio = Float.MIN_VALUE;
        ITask mostSwitches = null;
        
        System.out.println("--");
        System.out.println("InputMethodSwitching:");

        MultiEventTypeFilter keyboardInputFilter = new MultiEventTypeFilter(TextInput.class);
        MultiEventTypeFilter mouseInputFilter = new MultiEventTypeFilter(MouseInteraction.class);
        
        GenerateInstanceListVisitor v = new GenerateInstanceListVisitor();
        for (ITask task : taskModel.getTasks()) {            
            Collection<List<IEventTaskInstance>> instanceList = v.generateInstanceList(task);
            
            int instances = instanceList.size();
            int switches = 0;
            
            for (List<IEventTaskInstance> list : instanceList) {
                InputState current = InputState.unknown;

                for (IEventTaskInstance event : list) {
                    if ( keyboardInputFilter.matchesType(event.getEvent().getType()) ) {
                        switch (current)
                        {
                            case mouse:
                                switches++;
                            case unknown:
                                current = InputState.keyboard;
                            default:
                                break;
                        }
                                                
                    } else if (mouseInputFilter.matchesType(event.getEvent().getType())) {
                        switch (current)
                        {
                            case keyboard:
                                switches++;
                            case unknown:
                                current = InputState.mouse;
                            default:
                                break;
                        }                        
                    }
                }
            }
            
            float ratio = (float)switches / instances;
            if (ratio > maxRatio) {
                maxRatio = ratio;
                mostSwitches = task;
            }
        }

        System.out.printf("Most Switches in %s [%f]\n", mostSwitches.getDescription(), maxRatio);
        System.out.println("Finished InputMethodSwitching");
        return present;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.autoquest.usability.rules.UsabilityRule#callEvaluationMetho(de.ugoe.cs.autoquest
     * .usability.EvaluationMethodCaller)
     */
    @Override
    public Optional<UsabilityProblemDescription> callEvaluationMethod(EvaluationMethodCaller evaluationMethodCaller)
    {
        return evaluationMethodCaller.check(this);
    }
}
