//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.metrics;

import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.absent;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.highRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.infoRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.lowRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.mediumRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.present;
import static org.fest.assertions.api.Assertions.assertThat;

import org.junit.Test;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.testutil.GenerateTaskModelUtil;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputEntryRepetitionsEvaluatorTest {

    @Test
    public void should_return_no_recommendation() {
        // Given
        String spec = "UserSession {" +
        		      "  Sequence seq1 {" + 
        		      "    TextInput t1 () {}" + 
        		      "  }" +
        		      "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(absent());
    }

    @Test
    public void should_return_recommendation_with_info_severity_level() {
        // Given
        String spec = "UserSession {" +
        			  "  Sequence seq1 {" + 
        		      "    TextInput t1 (a b c) {}" +
        			  "    Sequence seq2 {" + 
        		      "      TextInput t2 (a) {}" + 
        			  "      TextInput t3 (d) {}" + 
        		      "      TextInput t4 (e) {}" + 
        			  "    }" +
        			  "  }" + 
        		      "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(infoRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_low_severity_level() {
        // Given
        String spec = "UserSession {" +
        			  "  Sequence seq1 {" + 
        			  "    TextInput t1 (a b c) {}" + 
        			  "    Sequence seq3 {" + 
        			  "      TextInput t2 (a) {}" + 
        			  "      TextInput t3 (b) {}" + 
        			  "      TextInput t4 (c) {}" + 
        			  "    }" +
        			  "  }" + 
        			  "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(lowRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_medium_severity_level() {
        // Given
        String spec = "UserSession {" +
        		      "  Sequence seq1 {" + 
        			  "    TextInput t1 (a b c d e f g h i j k l m) {}" + 
        		      "    Sequence seq2 {" + 
        			  "      TextInput t2 (a) {}" + 
        		      "      TextInput t3 (b) {}" + 
        			  "      TextInput t4 (c) {}" + 
        		      "      TextInput t5 (d) {}" + 
        			  "    }" + 
        		      "    Iteration iter1 {" + 
        			  "      TextInput t6 (e) {}" + 
        		      "    }" + 
        			  "    TextInput t7 (a) {}" + 
        		      "    Selection sel1 {" + 
        			  "      TextInput t8 (b) {}" +
        			  "    }" +
        		      "    Selection sel1 {" + 
        			  "      TextInput t8 (c) {}" +
        			  "    }" +
        		      "    Selection sel1 {" + 
        			  "      TextInput t8 (d) {}" +
        			  "    }" +
        		      "    Selection sel1 {" + 
        			  "      TextInput t8 (e) {}" +
        			  "    }" +
        		      "    Sequence seq3 {" + 
        			  "      TextInput t9 (a) {}" + 
        		      "      Sequence seq4 {" + 
        			  "        TextInput t10 (b) {}" + 
        		      "        TextInput t11 (c) {}" + 
        			  "        TextInput t12 (d) {}" + 
        		      "        TextInput t13 (e) {}" + 
        			  "      }" + 
        		      "    }" + 
        			  "    TextInput t14 (a) {}" +
        			  "  }" + 
        		      "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(mediumRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_high_severity_level() {
        // Given
        String spec = "UserSession {" +
		  		      "  Sequence seq1 {" + 
		  			  "    TextInput t1 (a b c) {}" + 
		  		      "    Sequence seq2 {" + 
		  			  "      TextInput t2 (a) {}" + 
		  		      "      TextInput t3 (b) {}" + 
		  			  "      TextInput t4 (c) {}" + 
		  		      "      TextInput t5 (a) {}" + 
		  			  "    }" + 
		  		      "    Iteration iter1 {" + 
		  			  "      TextInput t6 (a) {}" + 
		  		      "    }" + 
		  			  "    TextInput t7 (a) {}" + 
		  		      "    Selection sel1 {" + 
		  			  "      TextInput t8 (b c) {}" +
		  			  "    }" +
		  		      "    Selection sel1 {" + 
		  			  "      TextInput t8 (a) {}" +
		  			  "    }" +
		  		      "    Selection sel1 {" + 
		  			  "      TextInput t8 (a c) {}" +
		  			  "    }" +
		  		      "    Selection sel1 {" + 
		  			  "      TextInput t8 (b a) {}" +
		  			  "    }" +
		  		      "    Sequence seq3 {" + 
		  			  "      TextInput t9 (b c) {}" + 
		  		      "      Sequence seq4 {" + 
		  			  "        TextInput t10 (d a c) {}" + 
		  		      "        TextInput t11 (b b b a) {}" + 
		  			  "        TextInput t12 (a a c c) {}" + 
		  		      "        TextInput t13 (b b a) {}" + 
		  			  "      }" + 
		  		      "    }" + 
		  			  "    TextInput t14 (a) {}" +
		  			  "  }" + 
		  		      "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(highRecommendationSeverityLevel());
    }
}
