//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.httpmonitor;

import java.util.Arrays;

import de.ugoe.cs.autoquest.httpmonitor.proxy.HttpMonitoringProxy;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.TextConsole;

/**
 * <p>
 * implements the main method to start the {@link HttpMonitor} or the {@link HttpMonitoringProxy}.
 * Which one is to be started is determined by the first command line argument. If this is set to
 * "proxy" then the proxy is started. If this is set to "monitor" or left out then the monitor is
 * started.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class Runner {

    /**
     * <p>
     * Main method of the application.
     * </p>
     * 
     * @param args command line arguments
     */
    public static void main(String[] args) {
        new TextConsole();
        
        String type;
        if (args.length > 0) {
            type = args[0];
        }
        else {
            type = "monitor";
        }
        
        HttpMonitorComponent component;
        String[] remainingArgs = Arrays.copyOfRange(args, 1, args.length);
        
        if ("monitor".equalsIgnoreCase(type)) {
            component = new HttpMonitor(remainingArgs);
        }
        else if ("proxy".equalsIgnoreCase(type)) {
            component = new HttpMonitoringProxy(remainingArgs);
        }
        else {
            throw new IllegalArgumentException("unknown type of server requested: " + type);
        }
        
        try {
            component.init();
        }
        catch (HttpMonitorException e) {
            Console.printerrln("could not initialize HTTP " + type + " server: " + e.getMessage());
            Console.logException(e);
        }
        
        try {
            component.start();
        }
        catch (HttpMonitorException e) {
            Console.printerrln("could not start HTTP " + type + " server: " + e.getMessage());
            Console.logException(e);
        }
    }
}
