//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.http.eventcore;

import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.plugin.http.HTTPUtils;

/**
 * <p>
 * A simplified SOAP event type that only contains the name of the called method and the name of the
 * service.
 * </p>
 * 
 * @author Steffen Herbold
 */
public class SimpleSOAPEventType implements IEventType {

    /**  */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * the SOAP method called in this request
     * </p>
     */
    private final String calledMethod;

    /**
     * <p>
     * the name of the service; this is either the path or, if a path map is available
     * </p>
     */
    private final String serviceName;
    
    /**
     * <p>
     * the name of the client; this is either the host/ip and port of the sender or, if a path
     * map is available, a human readable name that may be based also on the receiver
     * </p>
     */
    private final String clientName;

    /**
     * <p>
     * Constructor. Creates a new SimpleSOAPEventType.
     * </p>
     * 
     */
    public SimpleSOAPEventType(String calledMethod, String serviceName, String clientName) {
        if (calledMethod == null) {
            throw new IllegalArgumentException("called method must not be null");
        }
        if (serviceName == null) {
            throw new IllegalArgumentException("serviceName must not be null");
        }
        if (clientName == null) {
            throw new IllegalArgumentException("clientName must not be null");
        }
        this.calledMethod = calledMethod;
        this.serviceName = serviceName;
        this.clientName = clientName;
    }

    /**
     * <p>
     * the SOAP method called in this request
     * </p>
     * 
     * @return the SOAP method called in this request
     */
    public String getCalledMethod() {
        return calledMethod;
    }

    /**
     * <p>
     * the name of the service called in this request
     * </p>
     * 
     * @return the name of the service called in this request
     */
    public String getServiceName() {
        return serviceName;
    }
    
    /**
     * <p>
     * the name of the client calling in this request
     * </p>
     *
     * @return the name of the client calling in this request
     */
    public String getClientName() {
        return clientName;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.eventcore.IEventType#getName()
     */
    @Override
    public String getName() {
        return "(" + serviceName + ", " + calledMethod + ")";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.http.eventcore.HTTPEventType#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        else if (obj instanceof SimpleSOAPEventType) {
            return HTTPUtils.equals(calledMethod, ((SimpleSOAPEventType) obj).calledMethod) &&
                HTTPUtils.equals(serviceName, ((SimpleSOAPEventType) obj).serviceName);
        }
        else {
            return false;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.http.eventcore.HTTPEventType#hashCode()
     */
    @Override
    public int hashCode() {
        int hashCode = calledMethod.hashCode() + serviceName.hashCode();
        return hashCode;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "SimpleSOAPEventType(" + serviceName + ", " + calledMethod + ")";
    }
}
