//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.http;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SOAPEventType;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SimpleSOAPEventType;
import de.ugoe.cs.autoquest.plugin.http.logdata.Address;

/**
 * <p>
 * contains convenience methods used for processing HTTP exchanges
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTTPUtils {

    /**
     * <p>
     * converts an address to a simple string containing either host or ip and the port number if
     * any.
     * </p>
     * 
     * @param address
     *            the address to convert
     * 
     * @return either "host:port" or "ip:port" or "host" or "ip" or "port" or null
     */
    public static String toString(Address address) {
        if (address != null) {
            StringBuffer buffer = new StringBuffer();
            String prefix = "";
            if (address.getHost() != null) {
                buffer.append(address.getHost());
                prefix = ":";
            }
            else if (address.getIp() != null) {
                buffer.append(prefix);
                buffer.append(address.getIp());
                prefix = ":";
            }

            if (address.getPort() != null) {
                buffer.append(prefix);
                buffer.append(address.getPort());
            }

            if (buffer.length() > 0) {
                return buffer.toString();
            }
        }

        return null;
    }

    /**
     * <p>
     * compares two addresses and returns true, if they are equal and false else. The addresses are
     * equal, if either the ip-addresses and the ports match or the host names and the ports match.
     * </p>
     * 
     * @param address1
     *            the first address to compare
     * @param address2
     *            the second address to compare
     * 
     * @return as described
     */
    public static boolean equals(Address address1, Address address2) {
        if (address1 == null) {
            return address2 == null;
        }
        else if (address2 == null) {
            return false;
        }

        if (!equals(address1.getPort(), address2.getPort())) {
            return false;
        }

        if (address1.getIp() != null) {
            return equals(address1.getIp(), address2.getIp());
        }
        else {
            return equals(address1.getHost(), address2.getHost());
        }
    }

    /**
     * <p>
     * convenience method to compare to objects. They are considered equal if they both are null, or
     * if their equals method returns true.
     * </p>
     * 
     * @param object1
     *            the first object to compare
     * @param object2
     *            the second object to compare
     * 
     * @return as described
     */
    public static <T> boolean equals(T object1, T object2) {
        if (object1 == null) {
            return object2 == null;
        }
        else {
            return object1.equals(object2);
        }
    }

    /**
     * <p>
     * Replaces events with a {@link SOAPEventType} with new events of {@link SimpleSOAPEventType}
     * and no target.
     * </p>
     * 
     * @param sequence
     *            sequence where the events are replaced
     * @param keepOnlySOAPEvents
     *            if true, all events that are not of SOAPEventType or SimpleSOAPEventType are
     *            removed
     * @return sequence with {@link SimpleSOAPEventType}s instead of {@link SOAPEventType}s
     */
    public static List<Event> convertToSimpleSOAPEvent(List<Event> sequence,
                                                       boolean keepOnlySOAPEvents)
    {
        List<Event> newSequence = null;
        if (sequence != null) {
            newSequence = new LinkedList<>();
            for (Event event : sequence) {
                if (event.getType() instanceof SOAPEventType) {
                    SOAPEventType eventType = (SOAPEventType) event.getType();
                    newSequence.add(new Event(new SimpleSOAPEventType(eventType.getCalledMethod(),
                                                                      eventType.getServiceName(),
                                                                      eventType.getClientName())));
                }
                else {
                    if (!keepOnlySOAPEvents || event.getType() instanceof SimpleSOAPEventType) {
                        newSequence.add(event);
                    }
                }
            }
        }
        return newSequence;
    }
    
    /**
     * <p>
     * Removes all non SOAP events from a sequence. Warning: this change is performed in-place!
     * </p>
     * 
     * @param sequence
     *            sequence where the events are replaced
     */
    public static void removeNonSOAPEvents(List<Event> sequence)
    {
        for (Iterator<Event> eventIter = sequence.iterator(); eventIter.hasNext();) {
            Event event = eventIter.next();
            if (!(event.getType() instanceof SOAPEventType)) {
                eventIter.remove();
            }
        }
    }

    /**
     * <p>
     * prevent instantiation
     * </p>
     */
    private HTTPUtils() {}

}
